const express = require("express");
const mongoose = require("mongoose");
const { taskRoutes } = require("./modules/tasks/task.routes");
const { authRoutes } = require("./modules/users/auth.routes");
const { userRoutes } = require("./modules/users/user.routes");
const { blogRoutes } = require("./modules/blogs/blog.routes");
const { newsRoutes } = require("./modules/news/news.routes");
const {
	newsletterRoutes,
} = require("./modules/newsletter/newsletterSubscriber.routes");
const problemReportRoutes = require("./modules/problemReports/problemReport.routes");
const voteRoutes = require("./modules/vote/vote.routes");
const productRoutes = require("./modules/products/product.routes");
const dbeProgramRoutes = require("./modules/dbe/dbeProgram.routes");
const dbeProgramAttendeeRoutes = require("./modules/dbe/dbeProgramAttendee.routes");
const pushSubscriptionRoutes = require("./modules/notifications/pushSubscription.routes");
const emailRoutes = require("./modules/notifications/email.routes");
const {
	failedPurchaseRoutes,
} = require("./modules/failedPurchase/failedPurchase.routes");
const {
	pancakeswapRoutes,
} = require("./modules/pancakeswapTransactions/pancakeswapTransaction.routes");
const { adminRoutes } = require("./modules/admin/admin.routes");
const { purchaseRoutes } = require("./modules/purchase/purchase.routes");
const { cacheService } = require("./core/cache/cache.service");
const { schedulerService } = require("./core/scheduler/scheduler.service");
const { getWorkQueue } = require("./queues/job.queue");
const { rankRoutes } = require("./modules/ranks/rank.routes");
const { rankSalaryRoutes } = require("./modules/ranks/rankSalary.routes");
const { walletRoutes } = require("./modules/wallets/wallet.routes");
const claimRequestRoutes = require("./modules/claims/claimRequest.routes");
const globalConfigRoutes = require("./modules/config/globalConfig.routes");
const { bonusTransactionRoutes } = require("./modules/bonuses/bonusTransaction.routes");
const { bonusDistributionRoutes } = require("./modules/bonuses/bonusDistribution.routes");
const cronJobRoutes = require("./modules/system/cronJob.routes");
const tokenStatsRoutes = require("./modules/auth/tokenStats.routes");
const { userPackageRoutes } = require("./modules/userPackages/userPackage.routes");
const { withdrawalRoutes } = require("./modules/withdrawals/withdrawal.routes");

const router = express.Router();

// Health check
router.get("/health", (req, res) => {
	res.json({
		status: "ok",
		timestamp: Date.now(),
		uptime: process.uptime(),
	});
});

// Readiness check
router.get("/ready", async (req, res) => {
	const checks = {
		mongodb: false,
		cache: false,
	};

	try {
		checks.mongodb = mongoose.connection.readyState === 1;
		await cacheService.set("health-check", true, 5);
		checks.cache = await cacheService.exists("health-check");
	} catch (err) {
		// continue
	}

	const allHealthy = Object.values(checks).every(Boolean);
	res.status(allHealthy ? 200 : 503).json({
		status: allHealthy ? "ready" : "degraded",
		timestamp: Date.now(),
		checks,
	});
});

// System metrics
router.get("/metrics", (req, res) => {
	const mem = process.memoryUsage();
	const queue = getWorkQueue();

	res.json({
		memory: {
			heapUsed: Math.round(mem.heapUsed / 1024 / 1024),
			heapTotal: Math.round(mem.heapTotal / 1024 / 1024),
			rss: Math.round(mem.rss / 1024 / 1024),
		},
		uptime: Math.round(process.uptime()),
		cache: cacheService.getStats(),
		queue: queue.getStats(),
		scheduler: schedulerService.getStatus(),
	});
});

// Authentication routes
router.use("/auth", authRoutes);

// User management routes
router.use("/users", userRoutes);

// Task routes
router.use("/tasks", taskRoutes);

// Blog routes
router.use("/blogs", blogRoutes);

// News routes
router.use("/news", newsRoutes);

// Newsletter routes
router.use("/newsletter", newsletterRoutes);

// Problem reports routes
router.use("/problem-reports", problemReportRoutes);

// Vote routes
router.use("/votes", voteRoutes);
// Product routes
router.use("/products", productRoutes);
// DBE Program routes
router.use("/dbe-programs", dbeProgramRoutes);
// DBE Program Attendee routes
router.use("/dbe-attendees", dbeProgramAttendeeRoutes);
// Push Subscription routes
router.use("/push-subscriptions", pushSubscriptionRoutes);
// Email routes
router.use("/email", emailRoutes);
// Failed Purchase routes
router.use("/failed-purchase", failedPurchaseRoutes);
// PancakeSwap Transaction routes
router.use("/pancakeswap", pancakeswapRoutes);
// Admin routes
// Purchase analytics routes
router.use("/purchase", purchaseRoutes);
router.use("/admin", adminRoutes);
// Ranks routes
router.use("/ranks", rankRoutes);
// Rank Salary routes
router.use("/rank-salary", rankSalaryRoutes);
// Wallet routes
router.use("/wallets", walletRoutes);
// Claim request routes
router.use("/claim-requests", claimRequestRoutes);
// Global Config routes
router.use("/global-config", globalConfigRoutes);
// Bonus Transaction routes
router.use("/bonus-transactions", bonusTransactionRoutes);
// Bonus Distribution routes (admin)
router.use("/bonus-distributions", bonusDistributionRoutes);
// Cron Job routes
router.use("/cron-jobs", cronJobRoutes);
// Token Stats routes
router.use("/token-stats", tokenStatsRoutes);
// User Packages routes
router.use("/user-packages", userPackageRoutes);
// Withdrawal routes
router.use("/withdrawals", withdrawalRoutes);

module.exports = { routes: router };
