require("dotenv").config();
const { app } = require("./app");
const { connectDb } = require("./config/db.config");
const { logger } = require("./core/logger/logger");
const { startWorker, stopWorker } = require("./workers/job.worker");
const {
	fetchAllTokenPrices,
} = require("./modules/config/tokenPriceFetcher.service");
const { cacheService } = require("./core/cache/cache.service");
const mongoose = require("mongoose");

const port = process.env.PORT || 5000;
let server;
let io;

// Graceful shutdown handler
const shutdown = async (signal) => {
	logger.info(`${signal} received, shutting down gracefully...`);

	// Stop worker scheduler
	try {
		await stopWorker();
	} catch (err) {
		logger.error("Error stopping worker", { error: err.message });
	}

	// Close Socket.IO
	if (io) {
		io.close(() => logger.info("Socket.IO server closed"));
	}

	// Stop accepting new connections
	if (server) {
		server.close(() => logger.info("HTTP server closed"));
	}

	// Close MongoDB connection
	try {
		await mongoose.connection.close();
		logger.info("MongoDB connection closed");
	} catch (err) {
		logger.error("Error closing MongoDB", { error: err.message });
	}

	process.exit(0);
};

process.on("SIGTERM", () => shutdown("SIGTERM"));
process.on("SIGINT", () => shutdown("SIGINT"));
process.on("uncaughtException", (err) => {
	logger.error("Uncaught exception", { error: err.message, stack: err.stack });
	process.exit(1);
});
process.on("unhandledRejection", (reason) => {
	logger.error("Unhandled rejection", { reason: String(reason) });
	process.exit(1);
});

(async () => {
	try {
		await connectDb();

		// Start job worker (auto-processes scheduled jobs every 5 min)
		await startWorker();

		// Start global config cron jobs (token price updates)
		const {
			initializeGlobalConfigCronJobs,
		} = require("./modules/config/globalConfig.cron");
		initializeGlobalConfigCronJobs();

		// Start main cron jobs (rank updates, bonus distribution, etc.)
		const { startCronJobs } = require("./core/scheduler/cronJob.service");
		startCronJobs();

		// Start API server
		server = app.listen(port, "0.0.0.0", () => {
			logger.info(`Server started on port ${port}`);
			logger.info("Job worker running with 5-minute scheduled task processing");
			logger.info(
				"Global config cron jobs initialized (token price auto-update)",
			);
		});

		// Initialize Socket.IO for real-time token prices
		io = require("socket.io")(server, {
			cors: {
				origin: process.env.CORS_ORIGIN || "*",
				methods: ["GET", "POST"],
			},
		});

		io.on("connection", async (socket) => {
			const username = socket.handshake.query.username;
			if (!username) {
				logger.warn("Connection attempt without username, disconnecting");
				socket.disconnect();
				return;
			}

			// Check if user is already connected
			const existingSocketId = await cacheService.get(
				`socket:user:${username}`,
			);
			if (existingSocketId) {
				// Disconnect the previous connection
				const previousSocket = io.sockets.sockets.get(existingSocketId);
				if (previousSocket) {
					logger.info(`Disconnecting previous connection for user ${username}`);
					previousSocket.disconnect(true);
				}
				// Remove the old cache entry
				await cacheService.delete(`socket:user:${username}`);
			}

			// Store the new connection
			await cacheService.set(`socket:user:${username}`, socket.id);
			logger.info(`User ${username} connected for token prices`);

			socket.on("disconnect", async () => {
				await cacheService.delete(`socket:user:${username}`);
				logger.info(`User ${username} disconnected from token prices`);
			});
		});

		// Emit token prices every 30 seconds
		setInterval(async () => {
			try {
				const prices = await fetchAllTokenPrices();
				if (prices) {
					io.emit("tokenPrices", prices);
				}
			} catch (error) {
				logger.error("Error emitting token prices:", error);
			}
		}, 30000);
	} catch (err) {
		logger.error("Failed to start server", { error: err.message });
		process.exit(1);
	}
})();
