/**
 * Contract utilities for wallet address validation and security
 */

const getApprovedReserveWallets = () => {
  const approvedWallets = process.env.APPROVED_RESERVE_WALLETS;
  
  if (!approvedWallets) {
    return [];
  }

  return approvedWallets
    .split(',')
    .map((addr) => addr.trim().toLowerCase())
    .filter((addr) => addr.length > 0);
};

const isApprovedReserveWallet = (address) => {
  if (!address) {
    return false;
  }

  const approvedWallets = getApprovedReserveWallets();
  const normalizedAddress = address.trim().toLowerCase();

  return approvedWallets.includes(normalizedAddress);
};

const getValidatedReserveWallet = (product) => {
  // Use product's reserve wallet address if available, otherwise use default
  const targetAddress =
    product?.reserveWalletAddress || process.env.RESERVE_WALLET_ADDRESS;

  if (!targetAddress) {
    throw new Error('No reserve wallet address configured');
  }

  // Validate against approved wallets
  if (!isApprovedReserveWallet(targetAddress)) {
    const approvedWallets = getApprovedReserveWallets();
    
    console.error(
      'Security violation: Attempted to use unapproved wallet address',
      {
        attempted: targetAddress,
        approved: approvedWallets,
        productId: product?._id,
        productName: product?.name,
      }
    );

    throw new Error(
      `Security Error: Wallet address ${targetAddress} is not approved for token transfers`
    );
  }

  return targetAddress;
};

const isValidWalletAddress = (address) => {
  if (!address || typeof address !== 'string') {
    return false;
  }

  // BSC address validation: 0x + 40 hex characters
  const addressRegex = /^0x[a-fA-F0-9]{40}$/;
  return addressRegex.test(address);
};

const normalizeWalletAddress = (address) => {
  if (!isValidWalletAddress(address)) {
    throw new Error(`Invalid wallet address format: ${address}`);
  }

  return address.trim().toLowerCase();
};

const compareWalletAddresses = (address1, address2) => {
  if (!address1 || !address2) {
    return false;
  }

  try {
    const normalized1 = normalizeWalletAddress(address1);
    const normalized2 = normalizeWalletAddress(address2);
    return normalized1 === normalized2;
  } catch (error) {
    return false;
  }
};

const getReserveWalletAddress = (product = null) => {
  return getValidatedReserveWallet(product);
};

const isValidTransactionHash = (txHash) => {
  if (!txHash || typeof txHash !== 'string') {
    return false;
  }

  // Transaction hash: 0x + 64 hex characters
  const txHashRegex = /^0x[a-fA-F0-9]{64}$/;
  return txHashRegex.test(txHash);
};

module.exports = {
  getApprovedReserveWallets,
  isApprovedReserveWallet,
  getValidatedReserveWallet,
  isValidWalletAddress,
  normalizeWalletAddress,
  compareWalletAddresses,
  getReserveWalletAddress,
  isValidTransactionHash,
};
