const { logger } = require("../core/logger/logger");
const {
	getTokenValue,
	getTotalUsersForPackageSync,
} = require("../modules/config/globalConfig.service");

async function convertUSDToNTE(usdAmount) {
	if (!Number.isFinite(usdAmount) || usdAmount < 0) {
		throw new Error(`Invalid USD amount for conversion: ${usdAmount}`);
	}
	const tokenPrice = await getTokenValue();
	if (!Number.isFinite(tokenPrice) || tokenPrice <= 0) {
		throw new Error(`Invalid token price from config: ${tokenPrice}`);
	}
	const usdWei = BigInt(Math.round(usdAmount * 1e18));
	const tokenWei = BigInt(Math.round(tokenPrice * 1e18));
	const nteWei = (usdWei * BigInt(1e18)) / tokenWei;
	const nteValue = Number(nteWei) / 1e18;
	if (!Number.isFinite(nteValue)) {
		throw new Error(
			`NTE calculation overflow: $${usdAmount} / $${tokenPrice} = ${nteValue}`,
		);
	}
	return nteValue;
}

async function convertNTEToUSD(nteAmount) {
	if (!Number.isFinite(nteAmount) || nteAmount < 0) {
		throw new Error(`Invalid NTE amount for conversion: ${nteAmount}`);
	}

	const tokenPrice = await getTokenValue();

	if (!Number.isFinite(tokenPrice) || tokenPrice <= 0) {
		throw new Error(`Invalid token price from config: ${tokenPrice}`);
	}

	const nteWei = BigInt(Math.round(nteAmount * 1e18));
	const tokenWei = BigInt(Math.round(tokenPrice * 1e18));
	const usdWei = (nteWei * tokenWei) / BigInt(1e18);
	const usdValue = Number(usdWei) / 1e18;

	if (!Number.isFinite(usdValue)) {
		throw new Error(
			`USD calculation overflow: ${nteAmount} NTE * $${tokenPrice} = ${usdValue}`,
		);
	}

	return usdValue;
}

async function formatNTEPrecision(nteAmount, options = {}) {
	const {
		calculationPrecision = 18,
		displayPrecision = "auto",
		showUnit = true,
		useShortForm = true,
	} = options;

	if (nteAmount === null || nteAmount === undefined || isNaN(nteAmount)) {
		return {
			calculation: 0,
			display: showUnit ? "0 NTE" : "0",
			raw: 0,
		};
	}

	const numValue = Number(nteAmount);
	const calculationValue = Number(numValue.toFixed(calculationPrecision));

	let displayValue;
	if (displayPrecision === "auto") {
		if (numValue >= 1000000) {
			displayValue = useShortForm
				? `${(numValue / 1000000).toFixed(2)}M`
				: numValue.toFixed(2);
		} else if (numValue >= 1000) {
			displayValue = useShortForm
				? `${(numValue / 1000).toFixed(2)}K`
				: numValue.toFixed(2);
		} else {
			displayValue = numValue.toFixed(2);
		}
	} else {
		displayValue = numValue.toFixed(displayPrecision);
	}

	const usdEquivalent = await convertNTEToUSD(calculationValue);

	return {
		calculation: calculationValue,
		display: showUnit ? `${displayValue} NTE` : displayValue,
		raw: numValue,
		usdEquivalent,
	};
}

async function formatUSDPrecision(usdAmount, options = {}) {
	const { precision = 2, showSymbol = true, useShortForm = true } = options;

	if (usdAmount === null || usdAmount === undefined || isNaN(usdAmount)) {
		return showSymbol ? "$0" : "0";
	}

	const numValue = Number(usdAmount);
	let displayValue;

	if (useShortForm && numValue >= 1000000) {
		displayValue = `${(numValue / 1000000).toFixed(2)}M`;
	} else if (useShortForm && numValue >= 1000) {
		displayValue = `${(numValue / 1000).toFixed(2)}K`;
	} else {
		displayValue = numValue.toFixed(precision);
	}

	return showSymbol ? `$${displayValue}` : displayValue;
}

async function formatNTEWithUSD(nteAmount, options = {}) {
	const nteFormatted = await formatNTEPrecision(nteAmount, options);
	const usdFormatted = await formatUSDPrecision(
		nteFormatted.usdEquivalent,
		options,
	);
	return `${nteFormatted.display} (${usdFormatted})`;
}

function getCurrentYearDays() {
	const currentYear = new Date().getFullYear();
	const isLeapYear =
		(currentYear % 4 === 0 && currentYear % 100 !== 0) ||
		currentYear % 400 === 0;
	return isLeapYear ? 366 : 365;
}

function calculateDailyPoolNTE(productData) {
	if (
		!productData?.userBonusCalculation?.annualBonus ||
		productData.userBonusCalculation.annualBonus <= 0
	)
		return 0;
	const annualBonusNTE = productData.userBonusCalculation.annualBonus;
	const lockPeriodValue =
		productData.userBonusCalculation.lockPeriod?.value || 2;
	if (lockPeriodValue <= 0) return 0;
	const daysPerYear = getCurrentYearDays();
	const annualBonusWei = BigInt(Math.round(annualBonusNTE * 1e18));
	const lockPeriodBig = BigInt(lockPeriodValue);
	const daysBig = BigInt(daysPerYear);
	const resultWei = annualBonusWei / lockPeriodBig / daysBig;
	const result = Number(resultWei) / 1e18;
	if (!Number.isFinite(result) || result < 0) {
		logger.error(
			`Invalid daily pool calculation: annualBonusNTE=${annualBonusNTE}, lockPeriod=${lockPeriodValue}, days=${daysPerYear}`,
		);
		return 0;
	}
	return result;
}

function calculateBonusPerUserNTE(productData) {
	const dailyPoolNTE = calculateDailyPoolNTE(productData);
	if (!dailyPoolNTE) return 0;
	const totalUsers = productData._id
		? getTotalUsersForPackageSync(productData._id)
		: 0;
	if (!totalUsers) return 0;
	const dailyPoolWei = BigInt(Math.round(dailyPoolNTE * 1e18));
	const totalUsersBig = BigInt(totalUsers);
	const resultWei = dailyPoolWei / totalUsersBig;
	const result = Number(resultWei) / 1e18;
	if (!Number.isFinite(result) || result < 0) {
		logger.error(
			`Invalid bonus per user calculation: dailyPoolNTE=${dailyPoolNTE}, totalUsers=${totalUsers}`,
		);
		return 0;
	}
	return result;
}

function calculateStakingBonusNTE(productData) {
	if (
		!productData?.userBonusCalculation?.aprPercent ||
		productData.userBonusCalculation.aprPercent <= 0 ||
		!productData?.priceNTE ||
		productData.priceNTE <= 0
	)
		return 0;
	const investmentAmountNTE = productData.priceNTE;
	const daysInYear = getCurrentYearDays();
	const aprPercent = productData.userBonusCalculation.aprPercent;
	const investmentWei = BigInt(Math.round(investmentAmountNTE * 1e18));
	const aprBig = BigInt(Math.round(aprPercent * 1e18));
	const hundredBig = BigInt(100 * 1e18);
	const daysBig = BigInt(daysInYear);
	const dailyBonusWei = (investmentWei * aprBig) / hundredBig / daysBig;
	const result = Number(dailyBonusWei) / 1e18;
	if (!Number.isFinite(result) || result < 0) {
		logger.error(
			`Invalid staking bonus calculation: investment=${investmentAmountNTE}, apr=${aprPercent}, days=${daysInYear}`,
		);
		return 0;
	}
	return result;
}

function calculateImmediateReferralBonusNTE(productData, generation = 1) {
	if (!productData?.priceNTE || !productData?.referralBonusImmediate) return 0;

	let percentage = 0;
	if (generation === 1) {
		percentage = productData.referralBonusImmediate.firstGenPercent;
	} else if (
		generation === 2 &&
		productData.referralBonusImmediate.secondGenPercent
	) {
		percentage = productData.referralBonusImmediate.secondGenPercent;
	} else if (
		generation > 2 &&
		productData.referralBonusImmediate.othersGenPercent
	) {
		percentage = productData.referralBonusImmediate.othersGenPercent;
	}

	if (!percentage || percentage < 0 || percentage > 100) return 0;

	// Precise math with BigInt to handle 18 decimals
	const priceWei = BigInt(Math.round(productData.priceNTE * 1e18));
	const percentageBigInt = BigInt(Math.round(percentage * 100));

	// Calculate bonus amount
	const bonusWei = (priceWei * percentageBigInt) / BigInt(10000);

	// Convert back to number
	const bonusNTE = Number(bonusWei) / 1e18;

	if (!Number.isFinite(bonusNTE) || bonusNTE < 0) {
		logger.error(
			`Invalid bonus calculation: priceNTE=${productData.priceNTE}, percentage=${percentage}`,
		);
		return 0;
	}

	return bonusNTE;
}

function calculateDailyReferralBonusNTE(generation = 1, productData) {
	if (!productData?.referralBonusCalculation?.perDayPercent) return 0;
	if (productData.packageType === "unlocked") {
		const userBonusNTE = calculateBonusPerUserNTE(productData);
		if (!userBonusNTE) return 0;
		let percentage = 0;
		if (
			generation === 1 &&
			productData.referralBonusCalculation.perDayPercent.firstGen
		) {
			percentage = productData.referralBonusCalculation.perDayPercent.firstGen;
		} else if (
			generation === 2 &&
			productData.referralBonusCalculation.perDayPercent.secondGen
		) {
			percentage = productData.referralBonusCalculation.perDayPercent.secondGen;
		} else if (
			generation > 2 &&
			productData.referralBonusCalculation.perDayPercent.othersGen
		) {
			percentage = productData.referralBonusCalculation.perDayPercent.othersGen;
		}
		if (!percentage) return 0;

		// Precise math with BigInt for 18 decimals
		const userBonusNTEWei = BigInt(Math.round(userBonusNTE * 1e18));
		const percentageBigInt = BigInt(Math.round(percentage * 100));

		// Calculate referral bonus
		const bonusWei = (userBonusNTEWei * percentageBigInt) / BigInt(10000);
		const bonusNTE = Number(bonusWei) / 1e18;

		if (!Number.isFinite(bonusNTE) || bonusNTE < 0) {
			logger.error(
				`Invalid daily referral bonus calculation: userBonusNTE=${userBonusNTE}, percentage=${percentage}`,
			);
			return 0;
		}
		return bonusNTE;
	} else {
		let baseAmount = 0;
		if (productData.userBonusCalculation?.aprPercent) {
			baseAmount = calculateStakingBonusNTE(productData);
		} else if (productData.userBonusCalculation?.bonusPercent) {
			baseAmount = calculateFlexibleStakingBonusNTE(productData);
		}
		if (!baseAmount) return 0;
		let percentage = 0;
		if (
			generation === 1 &&
			productData.referralBonusCalculation.perDayPercent.firstGen
		) {
			percentage = productData.referralBonusCalculation.perDayPercent.firstGen;
		} else if (
			generation === 2 &&
			productData.referralBonusCalculation.perDayPercent.secondGen
		) {
			percentage = productData.referralBonusCalculation.perDayPercent.secondGen;
		} else if (
			generation > 2 &&
			productData.referralBonusCalculation.perDayPercent.othersGen
		) {
			percentage = productData.referralBonusCalculation.perDayPercent.othersGen;
		}
		if (!percentage) return 0;

		// Precise math with BigInt for 18 decimals
		const baseAmountWei = BigInt(Math.round(baseAmount * 1e18));
		const percentageBigInt = BigInt(Math.round(percentage * 100));

		// Calculate bonus
		const bonusWei = (baseAmountWei * percentageBigInt) / BigInt(10000);
		const bonusNTE = Number(bonusWei) / 1e18;
		if (!Number.isFinite(bonusNTE) || bonusNTE < 0) {
			logger.error(
				`Invalid daily referral bonus calculation: baseAmount=${baseAmount}, percentage=${percentage}`,
			);
			return 0;
		}
		return bonusNTE;
	}
}

function calculateFlexibleStakingBonusNTE(productData) {
	if (
		!productData?.userBonusCalculation?.bonusPercent ||
		!productData?.priceNTE
	)
		return 0;

	// Using BigInt for precision
	const priceWei = BigInt(Math.round(productData.priceNTE * 1e18));
	const bonusPercentBigInt = BigInt(
		Math.round(productData.userBonusCalculation.bonusPercent * 100),
	);
	// Calculate bonus amount
	const bonusWei = (priceWei * bonusPercentBigInt) / BigInt(10000);
	const bonusNTE = Number(bonusWei) / 1e18;

	if (!Number.isFinite(bonusNTE) || bonusNTE < 0) {
		logger.error(
			`Invalid flexible staking bonus calculation: priceNTE=${productData.priceNTE}, bonusPercent=${productData.userBonusCalculation.bonusPercent}`,
		);
		return 0;
	}
	return bonusNTE;
}

async function calculateRankSalaryNTE(userRankData) {
	if (!userRankData?.monthlySalary || userRankData.monthlySalary <= 0) {
		return 0;
	}
	const salaryNTE = await convertUSDToNTE(userRankData.monthlySalary);
	return Number(salaryNTE.toString());
}

async function calculateTargetFillBonusNTE(productData, completedGroups) {
	if (
		!productData?.referralBonusTargetFill?.bonusPerGroupUSD ||
		productData.referralBonusTargetFill.bonusPerGroupUSD <= 0 ||
		completedGroups <= 0
	)
		return 0;
	const bonusPerGroupNTE = await convertUSDToNTE(
		productData.referralBonusTargetFill.bonusPerGroupUSD,
	);
	const bonusPerGroupWei = BigInt(Math.round(bonusPerGroupNTE * 1e18));
	const groupsBig = BigInt(completedGroups);
	const resultWei = bonusPerGroupWei * groupsBig;
	const result = Number(resultWei) / 1e18;
	if (!Number.isFinite(result) || result < 0) {
		logger.error(
			`Invalid target fill bonus calculation: bonusPerGroupUSD=${productData.referralBonusTargetFill.bonusPerGroupUSD}, groups=${completedGroups}`,
		);
		return 0;
	}
	return result;
}

module.exports = {
	convertUSDToNTE,
	convertNTEToUSD,
	formatNTEPrecision,
	formatUSDPrecision,
	formatNTEWithUSD,
	getCurrentYearDays,
	calculateDailyPoolNTE,
	calculateBonusPerUserNTE,
	calculateStakingBonusNTE,
	calculateImmediateReferralBonusNTE,
	calculateDailyReferralBonusNTE,
	calculateFlexibleStakingBonusNTE,
	calculateRankSalaryNTE,
	calculateTargetFillBonusNTE,
};
