const { logger } = require("../core/logger/logger");
const { User } = require("../modules/users/user.model");

class SystemNotificationService {
	/**
	 * Send a system notification to a single user via email and push notification
	 * Non-blocking: returns immediately and sends in background
	 */
	async sendToUser({
		username,
		email = null,
		subject,
		title,
		body,
		html = null,
		pushPayload = {},
		sendEmail = true,
		sendPush = true,
	}) {
		// Validate required fields
		if (!username) {
			logger.error("System notification: username is required");
			return {
				queued: false,
				error: "Username is required",
			};
		}

		if (!subject && !title) {
			logger.error("System notification: subject or title is required");
			return {
				queued: false,
				error: "Subject or title is required",
			};
		}

		if (!body) {
			logger.error("System notification: body is required");
			return {
				queued: false,
				error: "Body is required",
			};
		}

		logger.info(`System notification queued for ${username}`, {
			subject,
			sendEmail,
			sendPush,
		});

		// Execute in background without blocking
		setImmediate(() => {
			this._sendToUserBackground({
				username,
				email,
				subject,
				title,
				body,
				html,
				pushPayload,
				sendEmail,
				sendPush,
			}).catch((error) => {
				logger.error(`Background notification failed for ${username}:`, error);
			});
		});

		// Return immediately
		return {
			queued: true,
			username,
			message: "Notification queued for delivery",
		};
	}

	/**
	 * Internal method to send notification in background
	 */
	async _sendToUserBackground({
		username,
		email = null,
		subject,
		title,
		body,
		html = null,
		pushPayload = {},
		sendEmail = true,
		sendPush = true,
	}) {
		const results = {
			username,
			email: {
				sent: false,
				success: false,
				error: null,
			},
			push: {
				sent: false,
				success: false,
				sentCount: 0,
				failedCount: 0,
				error: null,
			},
		};

		try {
			// Fetch user email if not provided
			let userEmail = email;
			if (!userEmail && sendEmail) {
				try {
					const user = await User.findOne({ username }).select("email");
					if (user && user.email) {
						userEmail = user.email;
					} else {
						results.email.error = "User email not found";
						logger.warn(
							`System notification: User email not found for ${username}`,
						);
					}
				} catch (error) {
					results.email.error = `Failed to fetch user email: ${error.message}`;
					logger.error(
						`System notification: Failed to fetch user for ${username}:`,
						error,
					);
				}
			}

			// Send email notification
			if (sendEmail && userEmail) {
				try {
					results.email.sent = true;

					// Send email without database tracking
					const emailResult = await this._sendEmailDirect({
						to: userEmail,
						subject,
						text: body,
						html: html || `<p>${body}</p>`,
						sentBy: "system",
					});

					results.email.success = emailResult.success;
					results.email.messageId = emailResult.messageId;

					logger.info(`System notification email sent to ${username}`, {
						email: userEmail,
						subject,
					});
				} catch (error) {
					results.email.success = false;
					results.email.error = error.message;
					logger.error(
						`System notification email failed for ${username}:`,
						error,
					);
				}
			}

			// Send push notification
			if (sendPush) {
				try {
					results.push.sent = true;

					// Send push without database tracking
					const pushResult = await this._sendPushDirect(username, {
						title: title || subject,
						body,
						...pushPayload,
					});

					results.push.success = pushResult.success;
					results.push.sentCount = pushResult.sentCount || 0;
					results.push.failedCount = pushResult.failedCount || 0;
					results.push.totalSubscriptions = pushResult.totalSubscriptions || 0;

					logger.info(`System notification push sent to ${username}`, {
						sentCount: pushResult.sentCount,
						failedCount: pushResult.failedCount,
					});
				} catch (error) {
					results.push.success = false;
					results.push.error = error.message;
					logger.error(
						`System notification push failed for ${username}:`,
						error,
					);
				}
			}

			// Overall success if at least one channel succeeded
			results.overallSuccess = results.email.success || results.push.success;

			return results;
		} catch (error) {
			logger.error(`System notification failed for ${username}:`, error);
			throw error;
		}
	}

	/**
	 * Send a system notification to all admin users via email and push notification
	 * Non-blocking: returns immediately and sends in background
	 */
	async sendToAdmin({
		subject,
		title,
		body,
		html = null,
		pushPayload = {},
		sendEmail = true,
		sendPush = true,
	}) {
		// Validate required fields
		if (!subject && !title) {
			logger.error("System notification: subject or title is required");
			return {
				queued: false,
				error: "Subject or title is required",
			};
		}

		if (!body) {
			logger.error("System notification: body is required");
			return {
				queued: false,
				error: "Body is required",
			};
		}

		logger.info("System notification to admins queued", {
			subject,
			sendEmail,
			sendPush,
		});

		// Execute in background without blocking
		setImmediate(() => {
			this._sendToAdminBackground({
				subject,
				title,
				body,
				html,
				pushPayload,
				sendEmail,
				sendPush,
			}).catch((error) => {
				logger.error("Background notification to admins failed:", error);
			});
		});

		// Return immediately
		return {
			queued: true,
			message: "Notification queued for delivery to all admins",
		};
	}

	/**
	 * Internal method to send admin notifications in background
	 */
	async _sendToAdminBackground({
		subject,
		title,
		body,
		html = null,
		pushPayload = {},
		sendEmail = true,
		sendPush = true,
	}) {
		const aggregatedResults = {
			totalAdmins: 0,
			sent: 0,
			failed: 0,
			email: {
				sent: 0,
				failed: 0,
			},
			push: {
				sent: 0,
				failed: 0,
			},
			details: [],
		};

		try {
			// Fetch all admin users
			const adminUsers = await User.find({ role: "admin" }).select(
				"username email",
			);

			if (!adminUsers || adminUsers.length === 0) {
				logger.warn("System notification: No admin users found");
				return {
					...aggregatedResults,
					success: false,
					message: "No admin users found",
				};
			}

			aggregatedResults.totalAdmins = adminUsers.length;

			logger.info(
				`System notification: Sending to ${adminUsers.length} admin(s)`,
				{
					subject,
					sendEmail,
					sendPush,
				},
			);

			// Send to each admin
			const sendPromises = adminUsers.map(async (admin) => {
				try {
					const result = await this._sendToUserBackground({
						username: admin.username,
						email: admin.email,
						subject,
						title,
						body,
						html,
						pushPayload,
						sendEmail,
						sendPush,
					});

					// Track success/failure
					if (result.overallSuccess) {
						aggregatedResults.sent++;
					} else {
						aggregatedResults.failed++;
					}

					// Track email stats only if email sending was enabled
					if (sendEmail && result.email.sent) {
						if (result.email.success) {
							aggregatedResults.email.sent++;
						} else {
							aggregatedResults.email.failed++;
						}
					}

					// Track push stats only if push sending was enabled
					if (sendPush && result.push.sent) {
						if (result.push.success) {
							aggregatedResults.push.sent++;
						} else {
							aggregatedResults.push.failed++;
						}
					}

					aggregatedResults.details.push({
						username: admin.username,
						email: admin.email,
						success: result.overallSuccess,
						emailSent: sendEmail ? result.email.success : null,
						pushSent: sendPush ? result.push.success : null,
						pushDevices: sendPush ? result.push.sentCount || 0 : null,
					});

					return result;
				} catch (error) {
					aggregatedResults.failed++;
					aggregatedResults.details.push({
						username: admin.username,
						email: admin.email,
						success: false,
						error: error.message,
					});
					logger.error(
						`System notification failed for admin ${admin.username}:`,
						error,
					);
					return null;
				}
			});

			await Promise.all(sendPromises);

			aggregatedResults.success = aggregatedResults.sent > 0;
			aggregatedResults.message = `Sent to ${aggregatedResults.sent}/${aggregatedResults.totalAdmins} admin(s)`;

			logger.info("System notification to admins completed", {
				totalAdmins: aggregatedResults.totalAdmins,
				sent: aggregatedResults.sent,
				failed: aggregatedResults.failed,
				emailSent: aggregatedResults.email.sent,
				pushSent: aggregatedResults.push.sent,
			});

			return aggregatedResults;
		} catch (error) {
			logger.error("System notification to admins failed:", error);
			throw error;
		}
	}

	/**
	 * Send email directly without database tracking
	 */
	async _sendEmailDirect({ to, subject, text, html, sentBy }) {
		try {
			const nodemailer = require("nodemailer");

			// Get email config from environment
			const emailConfig = {
				host: process.env.SMTP_HOST || "smtp.gmail.com",
				port: parseInt(process.env.SMTP_PORT || "587"),
				secure: process.env.SMTP_SECURE === "true",
				auth: {
					user: process.env.SMTP_USER || "",
					pass: process.env.SMTP_PASS || "",
				},
			};

			logger.info("Email configuration loaded", {
				host: emailConfig.host,
				port: emailConfig.port,
				secure: emailConfig.secure,
				hasUser: !!emailConfig.auth.user,
				hasPass: !!emailConfig.auth.pass,
			});

			if (!emailConfig.auth.user || !emailConfig.auth.pass) {
				logger.error("Email service not configured - missing credentials", {
					SMTP_USER: process.env.SMTP_USER ? "SET" : "MISSING",
					SMTP_PASS: process.env.SMTP_PASS ? "SET" : "MISSING",
				});
				throw new Error(
					"Email service not configured - SMTP credentials missing",
				);
			}

			logger.info("Creating email transporter...");
			const transporter = nodemailer.createTransport(emailConfig);

			const mailOptions = {
				from: process.env.SMTP_FROM || emailConfig.auth.user,
				to,
				subject,
				text,
				html,
			};

			logger.info("Email options prepared", {
				from: mailOptions.from,
				to: mailOptions.to,
				subject: mailOptions.subject,
				hasHtml: !!mailOptions.html,
			});

			if (process.env.SEND_EMAILS !== "YES") {
				logger.info("Development mode - simulating email send");
				return {
					success: true,
					messageId: `dev-${Date.now()}@example.com`,
				};
			}

			logger.info("Sending email via SMTP...");
			const info = await transporter.sendMail(mailOptions);

			logger.info("Email sent successfully", {
				messageId: info.messageId,
				accepted: info.accepted,
				rejected: info.rejected,
			});

			return {
				success: true,
				messageId: info.messageId,
			};
		} catch (error) {
			logger.error("Email sending failed with detailed error:", {
				errorMessage: error.message,
				errorStack: error.stack,
				errorCode: error.code,
				command: error.command,
				response: error.response,
				responseCode: error.responseCode,
				to,
				subject,
			});
			throw error;
		}
	}

	/**
	 * Send push notification directly without database tracking
	 */
	async _sendPushDirect(username, payload) {
		try {
			const {
				PushSubscription,
			} = require("../modules/notifications/pushSubscription.model");
			const webpush = require("web-push");

			logger.info("Fetching push subscriptions", { username });

			// Get subscriptions
			const subscriptions = await PushSubscription.find({
				username,
				isActive: true,
			});

			logger.info("Push subscriptions found", {
				username,
				count: subscriptions.length,
			});

			if (subscriptions.length === 0) {
				logger.warn("No active push subscriptions found", { username });
				return {
					success: false,
					message: "No active subscriptions found",
					sentCount: 0,
					failedCount: 0,
					totalSubscriptions: 0,
				};
			}

			// Send to all devices
			const results = await Promise.allSettled(
				subscriptions.map(async (sub) => {
					try {
						await webpush.sendNotification(
							{
								endpoint: sub.endpoint,
								keys: {
									p256dh: sub.subscription.keys.p256dh,
									auth: sub.subscription.keys.auth,
								},
							},
							JSON.stringify(payload),
						);

						// Update lastUsed
						await PushSubscription.findByIdAndUpdate(sub._id, {
							lastUsed: new Date(),
						});

						return { success: true, endpoint: sub.endpoint };
					} catch (error) {
						// Remove expired subscriptions
						if (error.statusCode === 410) {
							await PushSubscription.findByIdAndDelete(sub._id);
						}
						return {
							success: false,
							endpoint: sub.endpoint,
							error: error.message,
						};
					}
				}),
			);

			const sentCount = results.filter((r) => r.value?.success).length;
			const failedCount = results.filter((r) => !r.value?.success).length;

			logger.info("Push notification send completed", {
				username,
				sentCount,
				failedCount,
				totalSubscriptions: subscriptions.length,
			});

			return {
				success: sentCount > 0,
				message: `Notification sent to ${sentCount}/${subscriptions.length} device(s)`,
				sentCount,
				failedCount,
				totalSubscriptions: subscriptions.length,
				results: results.map((r) => r.value),
			};
		} catch (error) {
			logger.error("Push notification sending failed with detailed error:", {
				errorMessage: error.message,
				errorStack: error.stack,
				username,
				payload,
			});
			throw error;
		}
	}
}

module.exports = new SystemNotificationService();
