const mongoose = require("mongoose");
const { logger } = require("../core/logger/logger");

let transactionsSupported = null;
let transactionCheckCompleted = false;

const checkTransactionSupport = async () => {
	if (transactionsSupported !== null && transactionCheckCompleted) {
		return transactionsSupported;
	}

	try {
		const admin = mongoose.connection.db.admin();
		const result = await admin.serverStatus();

		const isReplicaSet = result.repl && result.repl.setName;
		const isSharded = result.process === "mongos";

		transactionsSupported = isReplicaSet || isSharded;
		transactionCheckCompleted = true;

		if (transactionsSupported) {
			logger.info(
				"✅ MongoDB transactions supported (replica set or sharded cluster detected)",
				{
					replicaSet: isReplicaSet ? result.repl.setName : null,
					sharded: isSharded,
					environment: process.env.NODE_ENV,
				},
			);
		} else {
			const errorMsg =
				"MongoDB transactions NOT supported (standalone instance detected).";

			if (process.env.NODE_ENV === "production") {
				logger.error(errorMsg);
				logger.error(
					"Production deployment BLOCKED - replica set required for financial transaction integrity",
				);
				logger.error(
					"Configure MongoDB replica set with at least 3 nodes before production deployment",
				);
				throw new Error(
					"FATAL : MongoDB transactions are MANDATORY in production. Replica set configuration required. System halted.",
				);
			} else {
				logger.warn(
					errorMsg +
						" Running in development mode WITHOUT transaction guarantees.",
				);
				logger.warn("Data consistency NOT guaranteed - use only for testing!");
			}
		}

		return transactionsSupported;
	} catch (error) {
		if (error.message.includes("FATAL:")) {
			throw error;
		}

		logger.error("Failed to check MongoDB transaction support:", error);

		if (process.env.NODE_ENV === "production") {
			throw new Error(
				"FATAL: Cannot verify MongoDB transaction support in production. System halted.",
			);
		}

		transactionsSupported = false;
		transactionCheckCompleted = true;
		return false;
	}
};

const withTransaction = async (callback, options = {}) => {
	const { maxRetries = 3, retryDelay = 100 } = options;
	const supportsTransactions = await checkTransactionSupport();

	if (supportsTransactions) {
		let lastError;

		for (let attempt = 1; attempt <= maxRetries; attempt++) {
			const session = await mongoose.startSession();
			try {
				await session.startTransaction({
					readConcern: { level: "snapshot" },
					writeConcern: { w: "majority" },
					readPreference: "primary",
				});

				const result = await callback(session);
				await session.commitTransaction();

				if (attempt > 1) {
					logger.info(`Transaction succeeded on attempt ${attempt}`);
				}

				return result;
			} catch (error) {
				await session.abortTransaction();
				lastError = error;

				const isTransientError =
					error.hasErrorLabel &&
					error.hasErrorLabel("TransientTransactionError");
				const isWriteConflict = error.code === 112;

				if ((isTransientError || isWriteConflict) && attempt < maxRetries) {
					logger.warn(
						`Transaction failed (attempt ${attempt}/${maxRetries}), retrying...`,
						{
							error: error.message,
							code: error.code,
						},
					);
					await new Promise((resolve) =>
						setTimeout(resolve, retryDelay * attempt),
					);
				} else {
					throw error;
				}
			} finally {
				session.endSession();
			}
		}

		throw lastError;
	} else {
		logger.warn(
			"Executing operation without transaction - data consistency not guaranteed",
		);
		return await callback(null);
	}
};

const createSessionIfSupported = async () => {
	const supportsTransactions = await checkTransactionSupport();
	if (supportsTransactions) {
		return await mongoose.startSession();
	}
	return null;
};

const withSession = (query, session) => {
	return session ? query.session(session) : query;
};

module.exports = {
	checkTransactionSupport,
	withTransaction,
	createSessionIfSupported,
	withSession,
};
