const multer = require('multer');
const { writeFile, mkdir, unlink, access } = require('fs').promises;
const { join } = require('path');
const { ApiError } = require('./ApiError');
const { logger } = require('../core/logger/logger');

/**
 * Multer configuration for image uploads
 * Stores files in memory as buffers for flexible handling
 */
const multerConfig = multer({
  storage: multer.memoryStorage(),
  limits: {
    fileSize: 5 * 1024 * 1024, // 5MB limit
  },
  fileFilter: (req, file, cb) => {
    if (file.mimetype.startsWith('image/')) {
      cb(null, true);
    } else {
      cb(new ApiError(400, 'Only image files are allowed'), false);
    }
  },
});

/**
 * Upload Manager Class
 * Handles upload, replace, and delete operations for images
 */
class UploadManager {
  constructor(baseDir = 'uploads') {
    this.baseDir = baseDir;
  }

  /**
   * Generate unique filename
   * @param {string} originalName - Original file name
   * @param {string} prefix - Prefix for the filename (e.g., 'blog', 'user')
   * @returns {string} - Unique filename
   */
  generateFilename(originalName, prefix = 'file') {
    const timestamp = Date.now();
    const randomString = Math.random().toString(36).substring(2, 15);
    const extension = originalName.split('.').pop().toLowerCase();
    return `${prefix}_${timestamp}_${randomString}.${extension}`;
  }

  /**
   * Validate image file
   * @param {Object} file - Multer file object
   * @throws {ApiError} - If validation fails
   */
  validateImage(file) {
    if (!file) {
      throw new ApiError(400, 'No file provided');
    }

    if (!file.mimetype.startsWith('image/')) {
      throw new ApiError(400, 'Only image files are allowed');
    }

    const maxSize = 5 * 1024 * 1024; // 5MB
    if (file.size > maxSize) {
      throw new ApiError(400, 'File size must be less than 5MB');
    }

    // Validate file extension
    const allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'];
    const extension = file.originalname.split('.').pop().toLowerCase();
    if (!allowedExtensions.includes(extension)) {
      throw new ApiError(400, `File type .${extension} is not allowed. Allowed types: ${allowedExtensions.join(', ')}`);
    }

    return true;
  }

  /**
   * Upload new image
   * @param {Object} file - Multer file object (buffer-based)
   * @param {string} subDir - Subdirectory within base upload dir (e.g., 'blogs', 'avatars')
   * @param {string} prefix - Filename prefix
   * @returns {Promise<string>} - Relative path to uploaded file
   */
  async uploadImage(file, subDir = 'general', prefix = 'img') {
    this.validateImage(file);

    const filename = this.generateFilename(file.originalname, prefix);
    const uploadDir = join(process.cwd(), this.baseDir, subDir);
    
    // Ensure upload directory exists
    try {
      await mkdir(uploadDir, { recursive: true });
    } catch (error) {
      logger.warn('Error creating upload directory', { 
        uploadDir, 
        error: error.message 
      });
    }

    const filepath = join(uploadDir, filename);
    
    // Write file buffer to disk
    await writeFile(filepath, file.buffer);

    const relativePath = `/${this.baseDir}/${subDir}/${filename}`;
    
    logger.info('Image uploaded', { 
      filename, 
      size: file.size, 
      mimetype: file.mimetype,
      path: relativePath 
    });

    return relativePath;
  }

  /**
   * Replace existing image
   * Deletes old image and uploads new one
   * @param {Object} file - New multer file object
   * @param {string} oldImagePath - Path to old image to delete
   * @param {string} subDir - Subdirectory for new upload
   * @param {string} prefix - Filename prefix
   * @returns {Promise<string>} - Relative path to new uploaded file
   */
  async replaceImage(file, oldImagePath, subDir = 'general', prefix = 'img') {
    // Upload new image first
    const newImagePath = await this.uploadImage(file, subDir, prefix);
    
    // Delete old image (don't throw if delete fails)
    if (oldImagePath) {
      await this.deleteImage(oldImagePath, false);
    }

    return newImagePath;
  }

  /**
   * Delete image from filesystem
   * @param {string} imagePath - Relative path to image (e.g., '/uploads/blogs/image.jpg')
   * @param {boolean} throwOnError - Whether to throw error if delete fails
   * @returns {Promise<boolean>} - True if deleted, false otherwise
   */
  async deleteImage(imagePath, throwOnError = false) {
    if (!imagePath) {
      return false;
    }

    // Only delete files from our uploads directory
    if (!imagePath.startsWith(`/${this.baseDir}/`)) {
      logger.warn('Attempted to delete file outside uploads directory', { imagePath });
      return false;
    }

    try {
      // Remove leading /api if present (for Next.js compatibility)
      const cleanPath = imagePath.replace(/^\/api/, '');
      const filepath = join(process.cwd(), cleanPath);
      
      // Check if file exists before trying to delete
      try {
        await access(filepath);
      } catch {
        logger.warn('File does not exist, skipping delete', { imagePath });
        return false;
      }

      await unlink(filepath);
      logger.info('Image deleted', { imagePath });
      return true;
    } catch (error) {
      logger.error('Error deleting image', { 
        imagePath, 
        error: error.message 
      });
      
      if (throwOnError) {
        throw new ApiError(500, `Failed to delete image: ${error.message}`);
      }
      
      return false;
    }
  }

  /**
   * Delete multiple images
   * @param {Array<string>} imagePaths - Array of image paths to delete
   * @returns {Promise<Object>} - Object with success/failure counts
   */
  async deleteMultipleImages(imagePaths) {
    const results = {
      total: imagePaths.length,
      deleted: 0,
      failed: 0,
      errors: [],
    };

    for (const imagePath of imagePaths) {
      try {
        const deleted = await this.deleteImage(imagePath, false);
        if (deleted) {
          results.deleted++;
        } else {
          results.failed++;
        }
      } catch (error) {
        results.failed++;
        results.errors.push({ imagePath, error: error.message });
      }
    }

    return results;
  }

  /**
   * Check if file exists
   * @param {string} imagePath - Relative path to image
   * @returns {Promise<boolean>} - True if exists, false otherwise
   */
  async imageExists(imagePath) {
    if (!imagePath) return false;

    try {
      const cleanPath = imagePath.replace(/^\/api/, '');
      const filepath = join(process.cwd(), cleanPath);
      await access(filepath);
      return true;
    } catch {
      return false;
    }
  }

  /**
   * Get multer middleware for single file upload
   * @param {string} fieldName - Form field name (default: 'image')
   * @returns {Function} - Multer middleware
   */
  single(fieldName = 'image') {
    return multerConfig.single(fieldName);
  }

  /**
   * Get multer middleware for multiple file uploads
   * @param {string} fieldName - Form field name (default: 'images')
   * @param {number} maxCount - Maximum number of files (default: 10)
   * @returns {Function} - Multer middleware
   */
  array(fieldName = 'images', maxCount = 10) {
    return multerConfig.array(fieldName, maxCount);
  }

  /**
   * Get multer middleware for multiple fields
   * @param {Array} fields - Array of {name, maxCount} objects
   * @returns {Function} - Multer middleware
   */
  fields(fields) {
    return multerConfig.fields(fields);
  }
}

// Create singleton instance
const uploadManager = new UploadManager('uploads');

module.exports = { 
  uploadManager,
  UploadManager,
  multerConfig,
};
