const { ethers } = require("ethers");

const validators = {
	// Validate Ethereum address format
	isValidAddress(address) {
		return ethers.isAddress(address);
	},

	// Validate positive numeric amount
	isValidAmount(amount) {
		const num = Number(amount);
		return !isNaN(num) && isFinite(num) && num > 0;
	},

	// Validate amount with max limit 
	isValidAmountWithMax(amount, maxAmount) {
		if (!this.isValidAmount(amount)) return false;
		if (maxAmount && Number(amount) > maxAmount) return false;
		return true;
	},

	// Validate task status enum
	isValidTaskStatus(status) {
		return ["PENDING", "PROCESSING", "COMPLETED", "FAILED", "CANCELLED"].includes(status);
	},

	// Validate email format
	isValidEmail(email) {
		if (typeof email !== "string") return false;
		const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
		return emailRegex.test(email.toLowerCase());
	},

	// Validate positive integer (for pagination)
	isPositiveInt(value) {
		const num = parseInt(value, 10);
		return !isNaN(num) && num >= 0;
	},

	// Validate string is not empty and within length
	isValidString(str, maxLength = 255) {
		return (
			typeof str === "string" &&
			str.trim().length > 0 &&
			str.length <= maxLength
		);
	},

	// Validate asset symbol (alphanumeric, 1-10 chars)
	isValidAsset(asset) {
		return typeof asset === "string" && /^[A-Za-z0-9]{1,10}$/.test(asset);
	},

	// Sanitize string - remove potential injection characters
	sanitizeString(str) {
		if (typeof str !== "string") return "";
		return str.replace(/[${}]/g, "").trim();
	},

	// Parse and validate pagination params
	parsePagination(limit, skip) {
		const parsedLimit = Math.min(Math.max(parseInt(limit, 10) || 50, 1), 200);
		const parsedSkip = Math.max(parseInt(skip, 10) || 0, 0);
		return { limit: parsedLimit, skip: parsedSkip };
	},
};

module.exports = { validators };
