require("dotenv").config();
const mongoose = require("mongoose");
const { registerProcessor } = require("../queues/job.queue");
const { logger } = require("../core/logger/logger");
const { connectDb } = require("../config/db.config");
const { schedulerService } = require("../core/scheduler/scheduler.service");
const { taskService } = require("../modules/tasks/task.service");
const { checkTransactionSupport } = require("../utils/transaction.utils");
const {
	LOCK_TIMEOUTS,
	CLAIM_QUEUE_TIMING,
	CLAIM_LIMITS,
	TIME_CONSTANTS,
} = require("../utils/constants");
const { createWalletHash } = require("../modules/users/user.service");
const { isValidWalletAddress } = require("../utils/contractUtils");
const {
	WalletTransaction,
} = require("../modules/wallets/walletTransaction.model");
const {
	processImmediateReferralBonuses,
} = require("../modules/purchase/services/bonusDistribution.service");
const {
	PurchaseAnalytics,
} = require("../modules/purchase/purchaseAnalytics.model");
const { ClaimRequest } = require("../modules/claims/claimRequest.model");
const {
	claimRequestService,
} = require("../modules/claims/claimRequest.service");
const { User } = require("../modules/users/user.model");
const { walletService } = require("../modules/wallets/wallet.service");
const { auditLogService } = require("../modules/claims/auditLog.service");
const {
	AdminSystemSettings,
} = require("../modules/admin/adminSystemSettings.model");
const {
	processDynamicPricingForProduct,
} = require("../modules/purchase/services/dynamicPricing.service");
const {
	updatePackageOwnersCache,
} = require("../modules/purchase/services/packageOwners.service");
const {
	WithdrawalRequest,
} = require("../modules/withdrawals/withdrawalRequest.model");
const emailService = require("../modules/notifications/email.service");
const pushSubscriptionService = require("../modules/notifications/pushSubscription.service");
const systemNotificationService = require("../utils/system-notification.util");
const {
	generateEmailFromTemplate,
} = require("../templates/generateEmailFromTemplate");
const { getTokenValue } = require("../modules/config/globalConfig.service");

const SCHEDULER_INTERVAL_MS = 5 * 60 * 1000;
const CLAIM_APPROVAL_INTERVAL_MS = CLAIM_QUEUE_TIMING.APPROVAL_INTERVAL_MS;
const CLAIM_MIN_WAIT_TIME_MS = CLAIM_QUEUE_TIMING.MIN_WAIT_TIME_MS;
const QUEUE_CHECK_INTERVAL_MS = CLAIM_QUEUE_TIMING.QUEUE_CHECK_INTERVAL_MS;
const QUEUE_POSITION_UPDATE_INTERVAL_MS =
	CLAIM_QUEUE_TIMING.POSITION_UPDATE_INTERVAL_MS;
const MAX_CONSECUTIVE_FAILURES = 5;
const LOCK_TIMEOUT_MS = LOCK_TIMEOUTS.BALANCE_LOCK_MS;
const DISTRIBUTED_LOCK_TIMEOUT_MS = LOCK_TIMEOUTS.DISTRIBUTED_LOCK_MS;
const LOCK_HEARTBEAT_INTERVAL_MS = LOCK_TIMEOUTS.LOCK_HEARTBEAT_INTERVAL_MS;

let consecutiveFailures = 0;
let queuePaused = false;
let queuePausedReason = null;
let queuePausedAt = null;
let isProcessingClaim = false;
let currentProcessingClaimId = null;
let currentDistributedLockId = null;
let lockHeartbeatInterval = null;
let heartbeatFailureCount = 0;
const MAX_HEARTBEAT_FAILURES = 2;

// Process referral bonuses for a purchase
const processReferralBonusesJob = async (job) => {
	const { data } = job;
	const { purchaseId, username, packageIds, distributionIds, type, quantity } =
		data;

	try {
		logger.info(`Processing referral bonuses for ${quantity} packages...`);

		let bonusSuccessCount = 0;
		let bonusFailureCount = 0;

		const bonusBatchSize = Math.min(5, packageIds.length);
		const bonusBatches = Math.ceil(packageIds.length / bonusBatchSize);

		for (let batchIndex = 0; batchIndex < bonusBatches; batchIndex++) {
			const startIndex = batchIndex * bonusBatchSize;
			const endIndex = Math.min(startIndex + bonusBatchSize, packageIds.length);

			for (let i = startIndex; i < endIndex; i++) {
				try {
					await processImmediateReferralBonuses(
						username,
						packageIds[i],
						distributionIds[i],
						type,
					);
					bonusSuccessCount++;
				} catch (bonusError) {
					bonusFailureCount++;
					logger.error(
						`Error processing bonus for package ${i + 1}/${packageIds.length}:`,
						bonusError.message,
					);
				}
			}

			// Small delay between batches
			if (batchIndex < bonusBatches - 1) {
				await new Promise((resolve) => setTimeout(resolve, 25));
			}
		}

		logger.info(
			`Bonus processing completed: ${bonusSuccessCount} successful, ${bonusFailureCount} failed`,
		);

		logger.info(
			`Referral bonus processing completed for purchase ${purchaseId}`,
			{
				successful: bonusSuccessCount,
				failed: bonusFailureCount,
				total: packageIds.length,
				username,
			},
		);

		// Update purchase analytics with bonus results
		try {
			await PurchaseAnalytics.updateBonusResults(
				purchaseId,
				bonusSuccessCount,
				bonusFailureCount,
			);
			logger.info(
				`Purchase analytics updated with bonus results for ${purchaseId}`,
			);
		} catch (analyticsError) {
			logger.error(
				`Failed to update purchase analytics for ${purchaseId}:`,
				analyticsError.message,
			);
		}
	} catch (err) {
		logger.error(`Referral bonus job failed for ${purchaseId}`, {
			error: err.message,
			username,
			stack: err.stack,
		});
		throw err;
	}
};

// Process dynamic pricing update for a package
const processDynamicPricingJob = async (job) => {
	const { data } = job;
	const { packageId, eventType, purchaseId } = data;

	try {
		logger.info(
			`Processing dynamic pricing update for package ${packageId}...`,
			{
				eventType,
				purchaseId,
			},
		);

		const result = await processDynamicPricingForProduct(
			packageId,
			eventType,
			purchaseId,
		);

		logger.info(
			`DYNAMIC PRICING | Background update completed for package ${packageId}:`,
			result,
		);
	} catch (err) {
		logger.error(
			`DYNAMIC PRICING | Background update failed for package ${packageId}:`,
			err,
		);
		throw err;
	}
};

// Process package owners cache update
const processPackageOwnersCacheJob = async (job) => {
	const { data } = job;
	const { packageId, purchaseId } = data;

	try {
		logger.info(`Updating package owners cache for package ${packageId}...`, {
			purchaseId,
		});

		const result = await updatePackageOwnersCache(packageId, purchaseId);

		logger.info(
			`PACKAGE OWNERS | Cache update completed for package ${packageId}:`,
			result,
		);
	} catch (err) {
		logger.error(
			`PACKAGE OWNERS | Cache update failed for package ${packageId}:`,
			err,
		);
		throw err;
	}
};

// Process claim approval job - approves queued claims
const processClaimApprovalJob = async (job) => {
	const { data } = job;
	const { claimRequestId, taskId } = data;
	let lockedClaim = null;
	let distributedLock = null;

	logger.info(`[CLAIM JOB START] Processing claim approval job`, {
		claimRequestId,
		taskId,
		jobId: job.id,
	});

	try {
		// Check if queue is paused BEFORE any processing
		if (queuePaused) {
			logger.info(
				`CLAIM QUEUE | Queue is paused. Skipping execution without failing. Reason: ${queuePausedReason}`,
			);
			if (taskId)
				await taskService.markCompleted(
					taskId,
					`Skipped - Queue paused: ${queuePausedReason}`,
				);
			return { success: true, reason: "queue_paused_skipped" };
		}

		if (taskId) await taskService.markRunning(taskId);

		const crypto = require("crypto");
		const lockId = crypto.randomBytes(16).toString("hex");

		// Try to acquire distributed lock
		distributedLock = await ClaimRequest.findOneAndUpdate(
			{
				_id: claimRequestId,
				$or: [
					{ distributedLock: { $exists: false } },
					{ distributedLock: null },
					{ distributedLockExpiry: { $lt: new Date() } }, // Expired lock
				],
			},
			{
				$set: {
					distributedLock: lockId,
					distributedLockAcquiredAt: new Date(),
					distributedLockExpiry: new Date(
						Date.now() + DISTRIBUTED_LOCK_TIMEOUT_MS,
					),
					distributedLockWorker: "queue-worker",
					distributedLockHeartbeat: new Date(),
				},
			},
			{ new: true },
		);

		if (!distributedLock) {
			logger.warn(
				`CLAIM QUEUE | Failed to acquire distributed lock for ${claimRequestId} - another worker processing`,
			);
			return { success: false, reason: "distributed_lock_failed" };
		}

		logger.info(
			`Distributed lock acquired for ${claimRequestId} (lock ID: ${lockId})`,
		);

		currentDistributedLockId = lockId;
		currentProcessingClaimId = claimRequestId;

		lockHeartbeatInterval = setInterval(async () => {
			try {
				if (
					currentDistributedLockId === lockId &&
					currentProcessingClaimId === claimRequestId
				) {
					const refreshResult = await ClaimRequest.findOneAndUpdate(
						{ _id: claimRequestId, distributedLock: lockId },
						{
							$set: {
								distributedLockExpiry: new Date(
									Date.now() + DISTRIBUTED_LOCK_TIMEOUT_MS,
								),
								distributedLockHeartbeat: new Date(),
							},
						},
					);
					if (refreshResult) {
						logger.debug(
							`LOCK HEARTBEAT | Refreshed lock for ${claimRequestId}`,
						);
						heartbeatFailureCount = 0;
					} else {
						heartbeatFailureCount++;
						logger.warn(
							`LOCK HEARTBEAT | Failed to refresh lock for ${claimRequestId} - lock may have been stolen (failure ${heartbeatFailureCount}/${MAX_HEARTBEAT_FAILURES})`,
						);

						if (heartbeatFailureCount >= MAX_HEARTBEAT_FAILURES) {
							logger.error(
								`Aborting claim processing ${claimRequestId} - heartbeat failed ${heartbeatFailureCount} times`,
							);
							clearInterval(lockHeartbeatInterval);
							currentDistributedLockId = null;
							currentProcessingClaimId = null;
							throw new Error(
								`Heartbeat failure detected - aborting to prevent duplicate processing`,
							);
						}
					}
				}
			} catch (err) {
				heartbeatFailureCount++;
				logger.error(
					`LOCK HEARTBEAT | Error refreshing lock (failure ${heartbeatFailureCount}/${MAX_HEARTBEAT_FAILURES}):`,
					err.message,
				);

				if (heartbeatFailureCount >= MAX_HEARTBEAT_FAILURES) {
					logger.error(
						`Aborting claim processing - heartbeat error threshold exceeded`,
					);
					clearInterval(lockHeartbeatInterval);
					currentDistributedLockId = null;
					currentProcessingClaimId = null;
					throw err;
				}
			}
		}, LOCK_HEARTBEAT_INTERVAL_MS);

		logger.info(
			`CLAIM QUEUE | Starting approval process for ${claimRequestId}...`,
		);

		lockedClaim = await ClaimRequest.acquireProcessingLock(
			claimRequestId,
			"queue-worker",
		);

		if (!lockedClaim) {
			logger.warn(
				`CLAIM QUEUE | Failed to acquire processing lock for ${claimRequestId}`,
			);
			if (lockHeartbeatInterval) clearInterval(lockHeartbeatInterval);
			currentDistributedLockId = null;
			currentProcessingClaimId = null;
			await ClaimRequest.findOneAndUpdate(
				{ _id: claimRequestId, distributedLock: lockId },
				{
					$unset: {
						distributedLock: "",
						distributedLockAcquiredAt: "",
						distributedLockExpiry: "",
						distributedLockWorker: "",
						distributedLockHeartbeat: "",
					},
				},
			);
			return { success: false, reason: "lock_failed" };
		}

		const now = Date.now();
		const claimAge = now - new Date(lockedClaim.requestedAt).getTime();
		const minWaitTime =
			lockedClaim.amountNTE > 5000 ? 5 * 60 * 1000 : CLAIM_MIN_WAIT_TIME_MS; // 5 min for large amounts

		// Check if claim has waited minimum required time
		if (claimAge < minWaitTime) {
			const remainingWait = minWaitTime - claimAge;
			logger.info(
				`CLAIM QUEUE | Claim ${claimRequestId} needs to wait ${Math.ceil(
					remainingWait / 1000,
				)}s more (${Math.ceil(minWaitTime / 60000)} min required for ${
					lockedClaim.amountNTE
				} NTE)`,
			);
			await ClaimRequest.releaseProcessingLock(claimRequestId);
			// Clear heartbeat and release distributed lock
			if (lockHeartbeatInterval) clearInterval(lockHeartbeatInterval);
			currentDistributedLockId = null;
			currentProcessingClaimId = null;
			await ClaimRequest.findOneAndUpdate(
				{ _id: claimRequestId, distributedLock: lockId },
				{
					$unset: {
						distributedLock: "",
						distributedLockAcquiredAt: "",
						distributedLockExpiry: "",
						distributedLockWorker: "",
						distributedLockHeartbeat: "",
					},
				},
			);
			return { success: false, reason: "waiting_required_time" };
		}

		// Verify claim is still valid for processing
		if (lockedClaim.status !== "pending") {
			logger.info(
				`CLAIM QUEUE | Claim ${claimRequestId} already processed (status: ${lockedClaim.status})`,
			);
			await ClaimRequest.releaseProcessingLock(claimRequestId);
			// Clear heartbeat and release distributed lock
			if (lockHeartbeatInterval) clearInterval(lockHeartbeatInterval);
			currentDistributedLockId = null;
			currentProcessingClaimId = null;
			await ClaimRequest.findOneAndUpdate(
				{ _id: claimRequestId, distributedLock: lockId },
				{
					$unset: {
						distributedLock: "",
						distributedLockAcquiredAt: "",
						distributedLockExpiry: "",
						distributedLockWorker: "",
						distributedLockHeartbeat: "",
					},
				},
			);
			if (taskId) await taskService.markCompleted(taskId);
			return { success: true, reason: "already_processed" };
		}

		logger.info(
			`CLAIM QUEUE | Running pre-approval verification for ${claimRequestId}...`,
		);

		const verificationResult = await performPreApprovalVerification(
			lockedClaim,
		);

		if (!verificationResult.canProceed) {
			logger.warn(
				`CLAIM QUEUE | Pre-approval verification failed for ${claimRequestId}:`,
				verificationResult.reason,
			);

			await ClaimRequest.findByIdAndUpdate(claimRequestId, {
				$set: {
					preApprovalVerification: {
						verified: false,
						verifiedAt: new Date(),
						balanceAtVerification: verificationResult.currentBalance,
						riskLevel: verificationResult.riskLevel,
						checksPassedCount: verificationResult.checksPassedCount,
						checksTotalCount: verificationResult.checksTotalCount,
						warnings: verificationResult.warnings,
						errors: verificationResult.errors,
					},
				},
			});

			// Handle verification failure
			await ClaimRequest.markAutoApprovalFailed(
				claimRequestId,
				verificationResult.reason,
			);
			consecutiveFailures++;

			// Log audit
			await auditLogService.logSecurityViolation({
				type: "AUTO_APPROVAL_VERIFICATION_FAILED",
				username: lockedClaim.username,
				description: `Auto-approval verification failed: ${verificationResult.reason}`,
				severity:
					verificationResult.riskLevel === "HIGH" ? "CRITICAL" : "MEDIUM",
				clientIP: "queue-worker",
				userAgent: "queue-worker",
				metadata: {
					claimRequestId,
					verificationResult,
				},
			});

			isProcessingClaim = false;
			currentProcessingClaimId = null;
			if (taskId)
				await taskService.markFailed(taskId, verificationResult.reason);
			return { success: false, reason: verificationResult.reason };
		}

		// Update verification snapshot
		await ClaimRequest.findByIdAndUpdate(claimRequestId, {
			$set: {
				preApprovalVerification: {
					verified: true,
					verifiedAt: new Date(),
					balanceAtVerification: verificationResult.currentBalance,
					riskLevel: verificationResult.riskLevel,
					checksPassedCount: verificationResult.checksPassedCount,
					checksTotalCount: verificationResult.checksTotalCount,
					warnings: verificationResult.warnings,
					errors: [],
				},
			},
		});

		logger.info(
			`CLAIM QUEUE | Executing auto-approval for ${claimRequestId}...`,
		);

		const result = await claimRequestService.approveClaimRequest({
			claimRequestId: claimRequestId,
			masterPassword: "",
			notes: `Auto-approved via queue system. Verification passed at ${new Date().toISOString()}`,
			adminUsername: "system-queue",
			adminUserId: "system-queue",
			clientIP: "queue-worker",
			userAgent: "queue-worker",
			autoApproval: true,
		});

		await AdminSystemSettings.findOneAndUpdate(
			{ type: "system_config" },
			{
				$set: {
					lastClaimApprovalTime: new Date(),
					lastClaimApprovalBy: "system-queue",
				},
			},
			{ upsert: true },
		);
		consecutiveFailures = 0;

		if (lockHeartbeatInterval) {
			clearInterval(lockHeartbeatInterval);
			lockHeartbeatInterval = null;
		}
		currentDistributedLockId = null;
		currentProcessingClaimId = null;

		await ClaimRequest.findOneAndUpdate(
			{ _id: claimRequestId },
			{
				$set: {
					queueStatus: "completed",
					processingLock: false,
					processingLockedAt: null,
					processingLockedBy: null,
				},
				$unset: {
					distributedLock: "",
					distributedLockAcquiredAt: "",
					distributedLockExpiry: "",
					distributedLockWorker: "",
					distributedLockHeartbeat: "",
				},
			},
		);

		logger.info(`Auto-approval completed for ${claimRequestId}`, {
			success: result.success,
			username: lockedClaim.username,
			amountNTE: lockedClaim.amountNTE,
			transactionHash: result.transactionHash,
		});

		await auditLogService.logWithdrawalProcessing({
			claimRequestId: claimRequestId,
			username: lockedClaim.username,
			amountNTE: lockedClaim.amountNTE,
			toAddress: lockedClaim.walletAddress,
			status: "auto_approved",
			transactionHash: result.transactionHash,
		});

		if (taskId) await taskService.markCompleted(taskId);

		logger.info(`[CLAIM JOB SUCCESS] Claim approval completed successfully`, {
			claimRequestId,
			taskId,
			status: result.status || "approved",
			transactionHash: result.transactionHash,
		});

		return { success: true, result };
	} catch (err) {
		logger.error(`[CLAIM JOB ERROR] Claim approval job failed`, {
			claimRequestId,
			taskId,
			error: err.message,
			stack: err.stack,
		});

		logger.error(`Auto-approval failed for ${claimRequestId}`, {
			error: err.message,
			stack: err.stack,
		});

		if (lockHeartbeatInterval) {
			clearInterval(lockHeartbeatInterval);
			lockHeartbeatInterval = null;
		}
		currentDistributedLockId = null;
		currentProcessingClaimId = null;

		if (distributedLock) {
			await ClaimRequest.findOneAndUpdate(
				{
					_id: claimRequestId,
					distributedLock: distributedLock.distributedLock,
				},
				{
					$unset: {
						distributedLock: "",
						distributedLockAcquiredAt: "",
						distributedLockExpiry: "",
						distributedLockWorker: "",
						distributedLockHeartbeat: "",
					},
				},
			);
		}

		const isLockConflict =
			err.message &&
			(err.message.includes("currently being processed") ||
				err.message.includes("Failed to acquire lock") ||
				err.message.includes("lock_failed") ||
				err.message.includes("distributed_lock_failed"));

		if (isLockConflict) {
			logger.warn(
				`CLAIM QUEUE | Lock conflict detected - this is not a critical failure`,
			);

			if (taskId) await taskService.markFailed(taskId, err.message);
			return { success: false, reason: "lock_conflict" };
		}

		consecutiveFailures++;

		await ClaimRequest.markAutoApprovalFailed(claimRequestId, err.message);

		if (consecutiveFailures >= MAX_CONSECUTIVE_FAILURES) {
			queuePaused = true;
			queuePausedReason = `Auto-paused after ${consecutiveFailures} consecutive failures. Last error: ${err.message}`;
			queuePausedAt = new Date();
			logger.error(
				`CLAIM QUEUE | Queue auto-paused due to ${consecutiveFailures} consecutive failures`,
			);

			// Send critical notification to admins
			setImmediate(async () => {
				try {
					const emailBody = `
						<div style="padding: 20px; background-color: #7f1d1d; border-left: 4px solid #ef4444; margin: 20px 0;">
							<h2 style="color: #ef4444; margin: 0 0 15px 0; font-size: 20px;">🚨 CRITICAL: Claim Queue Auto-Paused</h2>
							<p style="color: #fca5a5; margin: 10px 0; font-size: 16px; font-weight: 600;">The automated claim approval queue has been paused due to consecutive failures.</p>
						</div>

						<div style="background-color: #060b18; padding: 15px; border-radius: 8px; margin: 20px 0;">
							<h3 style="color: #00e5ff; margin: 0 0 15px 0; font-size: 16px;">📊 Failure Details</h3>
							<table style="width: 100%; border-collapse: collapse;">
								<tr>
									<td style="padding: 8px; color: #94a3b8; border-bottom: 1px solid #1e293b;">Consecutive Failures:</td>
									<td style="padding: 8px; color: #ef4444; font-weight: 600; border-bottom: 1px solid #1e293b;">${consecutiveFailures} / ${MAX_CONSECUTIVE_FAILURES}</td>
								</tr>
								<tr>
									<td style="padding: 8px; color: #94a3b8; border-bottom: 1px solid #1e293b;">Failed Claim ID:</td>
									<td style="padding: 8px; color: #ffffff; font-family: 'Courier New', monospace; border-bottom: 1px solid #1e293b;">${claimRequestId}</td>
								</tr>
								<tr>
									<td style="padding: 8px; color: #94a3b8; border-bottom: 1px solid #1e293b;">Paused At:</td>
									<td style="padding: 8px; color: #ffffff; border-bottom: 1px solid #1e293b;">${queuePausedAt.toUTCString()}</td>
								</tr>
								<tr>
									<td style="padding: 8px; color: #94a3b8; border-bottom: 1px solid #1e293b;">Worker:</td>
									<td style="padding: 8px; color: #ffffff; border-bottom: 1px solid #1e293b;">queue-worker</td>
								</tr>
								<tr>
									<td style="padding: 8px; color: #94a3b8; vertical-align: top;">Last Error:</td>
									<td style="padding: 8px; color: #fca5a5; font-family: 'Courier New', monospace; word-break: break-word;">${
										err.message
									}</td>
								</tr>
							</table>
						</div>

						<div style="background-color: #7f1d1d; padding: 15px; border-radius: 8px; margin: 20px 0;">
							<h3 style="color: #ef4444; margin: 0 0 15px 0; font-size: 16px;">⚠️ Immediate Actions Required</h3>
							<ul style="color: #fca5a5; margin: 0; padding-left: 20px;">
								<li style="margin: 8px 0;">Investigate the root cause of the failures immediately</li>
								<li style="margin: 8px 0;">Review the failed claim request and error logs</li>
								<li style="margin: 8px 0;">Check system resources and database connectivity</li>
								<li style="margin: 8px 0;">Verify blockchain connection and smart contract status</li>
								<li style="margin: 8px 0;">Review pending claims queue for stuck requests</li>
								<li style="margin: 8px 0;">Resume queue manually after resolving issues</li>
							</ul>
						</div>

						<div style="background-color: #1e293b; padding: 15px; border-radius: 8px; margin: 20px 0;">
							<p style="color: #94a3b8; margin: 0; font-size: 14px;">
								<strong style="color: #f59e0b;">⚠️ Impact:</strong> All automated claim approvals are currently paused. Users may experience delays in withdrawal processing until the queue is manually resumed.
							</p>
						</div>

						<div style="text-align: center; margin: 30px 0 20px 0;">
							<a href="${
								process.env.ADMIN_PANEL_URL || "https://admin.nodemeta.io"
							}/claims" style="display: inline-block; padding: 12px 30px; background-color: #ef4444; color: #ffffff; text-decoration: none; border-radius: 6px; font-weight: 600;">View Claims Queue</a>
						</div>

						<p style="color: #64748b; font-size: 12px; margin-top: 30px; padding-top: 20px; border-top: 1px solid #1e293b;">
							This is an automated critical alert from the NodeMeta Queue Worker. Immediate administrator attention is required.
						</p>
					`;

					const htmlContent = generateEmailFromTemplate(emailBody);

					systemNotificationService.sendToAdmin({
						subject: `🚨 CRITICAL: Claim Queue Auto-Paused After ${consecutiveFailures} Failures`,
						title: "CRITICAL: Claim Queue Auto-Paused",
						body: `The automated claim approval queue has been paused due to ${consecutiveFailures} consecutive failures. Failed Claim ID: ${claimRequestId}. Last Error: ${err.message}. Immediate administrator attention required.`,
						html: htmlContent,
						pushPayload: {
							icon: "/icons/icon-192x192.png",
							badge: "/icons/badge-72x72.png",
							data: {
								url: "/admin/claim-request",
								failedClaimId: claimRequestId,
								status: "auto_paused",
							},
						},
						sendEmail: true,
						sendPush: true,
					});

					logger.info(
						"Critical admin notification sent for claim queue auto-pause",
					);
				} catch (notificationError) {
					logger.error(
						"Failed to send claim queue pause notification to admins:",
						{
							error: notificationError.message,
							stack: notificationError.stack,
						},
					);
				}
			});

			await auditLogService.logSecurityViolation({
				type: "CLAIM_QUEUE_AUTO_PAUSED",
				username: "system",
				description: queuePausedReason,
				severity: "CRITICAL",
				clientIP: "queue-worker",
				userAgent: "queue-worker",
			});
		}

		isProcessingClaim = false;
		currentProcessingClaimId = null;
		if (taskId) await taskService.markFailed(taskId, err.message);
		throw err;
	}
};

const performPreApprovalVerification = async (claim) => {
	logger.info(`[VERIFICATION START] Starting pre-approval verification`, {
		claimId: claim._id,
		username: claim.username,
		amount: claim.amountNTE,
	});

	const result = {
		canProceed: true,
		reason: null,
		riskLevel: "LOW",
		currentBalance: 0,
		checksPassedCount: 0,
		checksTotalCount: 0,
		warnings: [],
		errors: [],
	};

	const checks = [];

	try {
		// 1. Global Withdrawals Check
		checks.push({ name: "Global Withdrawals" });
		const systemConfig = await AdminSystemSettings.findOne({
			type: "system_config",
		});
		if (systemConfig && systemConfig.withdrawalsEnabled === false) {
			result.canProceed = false;
			result.reason = "Withdrawals are disabled system-wide";
			result.errors.push("Global withdrawals disabled");
			result.riskLevel = "CRITICAL";
			return result;
		}
		result.checksPassedCount++;

		// 2. User Status & Verification
		checks.push({ name: "User Status" });
		const user = await User.findOne({ username: claim.username });
		if (!user) {
			result.canProceed = false;
			result.reason = "User not found";
			result.errors.push("User account not found");
			return result;
		}
		if (user.isBlocked || !user.isActive) {
			result.canProceed = false;
			result.reason = "User account is blocked";
			result.errors.push("User account blocked");
			result.riskLevel = "CRITICAL";
			return result;
		}
		if (user.withdrawalsDisabled) {
			result.canProceed = false;
			result.reason = "Withdrawals disabled for user";
			result.errors.push("User withdrawals disabled");
			result.riskLevel = "CRITICAL";
			return result;
		}
		result.checksPassedCount++;

		// 3. Wallet Address Verification
		checks.push({ name: "Wallet Verification" });
		const requestedWalletHash = createWalletHash(claim.walletAddress);
		if (user.walletHash && user.walletHash !== requestedWalletHash) {
			result.canProceed = false;
			result.reason = "Wallet address does not match registered wallet";
			result.errors.push("Wallet address mismatch");
			result.riskLevel = "CRITICAL";
			return result;
		}
		if (!isValidWalletAddress(claim.walletAddress)) {
			result.canProceed = false;
			result.reason = "Invalid wallet address format";
			result.errors.push("Invalid wallet address format");
			result.riskLevel = "HIGH";
			return result;
		}
		result.checksPassedCount++;

		// 4. Account Age Check
		checks.push({ name: "Account Age" });
		const userAge = user.createdAt
			? Math.floor(
					(Date.now() - new Date(user.createdAt).getTime()) /
						TIME_CONSTANTS.ONE_DAY_MS,
			  )
			: 0;

		if (userAge < CLAIM_LIMITS.MIN_ACCOUNT_AGE_DAYS) {
			result.canProceed = false;
			result.reason = `Account is less than ${CLAIM_LIMITS.MIN_ACCOUNT_AGE_DAYS} day old`;
			result.errors.push("Account too new");
			result.riskLevel = "CRITICAL";
			return result;
		}

		if (userAge < CLAIM_LIMITS.NEW_ACCOUNT_WARNING_DAYS) {
			const successfulClaimsCount = await ClaimRequest.countDocuments({
				username: claim.username,
				_id: { $ne: claim._id },
				status: { $in: ["approved", "completed"] },
			});
			if (successfulClaimsCount === 0) {
				result.warnings.push(
					`Account is ${userAge} days old with no successful claims`,
				);
				result.riskLevel = "MEDIUM";
			}
		}
		result.checksPassedCount++;

		// 5. Balance Verification
		checks.push({ name: "Balance Verification" });
		const balanceResult = await walletService.getBalanceBreakdown(
			claim.username,
			true,
		);
		if (!balanceResult) {
			result.canProceed = false;
			result.reason = "Unable to retrieve balance";
			result.errors.push("Balance retrieval failed");
			return result;
		}

		// Check for negative balances
		const originalBreakdown = balanceResult.breakdown;
		const balanceCategories = [
			"core",
			"elite",
			"stake",
			"meta_pulse",
			"others",
			"total",
		];
		for (const category of balanceCategories) {
			if (originalBreakdown[category] < 0) {
				result.canProceed = false;
				result.reason = `Negative balance detected in ${category}`;
				result.errors.push(`Negative balance: ${category}`);
				result.riskLevel = "CRITICAL";
				return result;
			}
		}

		result.currentBalance = balanceResult.availableBreakdown.total;

		if (balanceResult.availableBreakdown.total < claim.amountNTE) {
			result.canProceed = false;
			result.reason = `Insufficient balance: ${balanceResult.availableBreakdown.total.toFixed(
				6,
			)} < ${claim.amountNTE}`;
			result.errors.push("Insufficient balance");
			result.riskLevel = "HIGH";
			return result;
		}

		if (user.balanceLocked === true) {
			const lockAge = user.balanceLockedAt
				? Math.floor(
						(Date.now() - new Date(user.balanceLockedAt).getTime()) / 1000,
				  )
				: 0;
			if (lockAge < 120) {
				result.canProceed = false;
				result.reason = `Balance locked by ${user.balanceLockedBy} (${lockAge}s ago)`;
				result.errors.push("Balance currently locked");
				result.riskLevel = "MEDIUM";
				return result;
			} else {
				result.warnings.push(`Stale balance lock detected (${lockAge}s old)`);
			}
		}
		result.checksPassedCount++;

		// 6. Balance Consistency Check
		checks.push({ name: "Balance Consistency" });
		if (user.walletBalance !== undefined) {
			const difference = Math.abs(user.walletBalance - originalBreakdown.total);
			if (difference > 0.000001) {
				result.errors.push(`Balance mismatch: ${difference.toFixed(6)} NTE`);
			}
		}

		const balanceDrop = claim.walletBalanceBefore - user.walletBalance;
		const balanceDropPercent =
			claim.walletBalanceBefore > 0
				? (balanceDrop / claim.walletBalanceBefore) * 100
				: 0;

		if (balanceDropPercent > 20 && balanceDrop > 10) {
			result.warnings.push(
				`Balance dropped ${balanceDropPercent.toFixed(1)}% since claim`,
			);
			result.riskLevel = "MEDIUM";
		}
		result.checksPassedCount++;

		// 7. Amount Validation
		checks.push({ name: "Amount Validation" });
		if (claim.amountNTE < CLAIM_LIMITS.MIN_AMOUNT_NTE) {
			result.canProceed = false;
			result.reason = `Amount below minimum ${CLAIM_LIMITS.MIN_AMOUNT_NTE} NTE`;
			result.errors.push("Amount below minimum threshold");
			return result;
		}
		const MAX_AUTO_APPROVE_AMOUNT = parseFloat(
			CLAIM_LIMITS.MAX_AUTO_APPROVE_NTE,
		);
		const currentNTEPrice = await getTokenValue();
		const usdValue = claim.amountNTE * (currentNTEPrice || 0.001);

		if (usdValue > MAX_AUTO_APPROVE_AMOUNT) {
			result.canProceed = false;
			result.reason = `Amount ${claim.amountNTE} exceeds auto-approval limit ${MAX_AUTO_APPROVE_AMOUNT}`;
			result.errors.push("Amount exceeds auto-approval limit");
			result.riskLevel = "HIGH";
			return result;
		}

		if (claim.amountNTE > user.walletBalance) {
			result.canProceed = false;
			result.reason = "Claim amount exceeds wallet balance";
			result.errors.push("Amount exceeds wallet balance");
			result.riskLevel = "HIGH";
			return result;
		}
		result.checksPassedCount++;

		// 8. Withdrawal Percentage Warning
		checks.push({ name: "Withdrawal Percentage" });
		const totalAvailableBalance = balanceResult.availableBreakdown.total;
		const claimPercentage =
			totalAvailableBalance > 0
				? (claim.amountNTE / totalAvailableBalance) * 100
				: 0;

		if (claimPercentage > 90) {
			result.warnings.push(
				`Withdrawing ${claimPercentage.toFixed(1)}% of total balance`,
			);
			result.riskLevel =
				result.riskLevel === "CRITICAL" ? "CRITICAL" : "MEDIUM";
		} else if (claimPercentage > 50) {
			result.warnings.push(
				`Withdrawing ${claimPercentage.toFixed(1)}% of total balance`,
			);
		}
		result.checksPassedCount++;

		// 9. Duplicate Claims Check
		checks.push({ name: "Duplicate Check" });
		const duplicateClaims = await ClaimRequest.countDocuments({
			username: claim.username,
			status: "pending",
			_id: { $ne: claim._id },
		});
		if (duplicateClaims > 0) {
			result.canProceed = false;
			result.reason = "User has other pending claims";
			result.errors.push("Duplicate pending claims found");
			result.riskLevel = "MEDIUM";
			return result;
		}
		result.checksPassedCount++;

		// 10. Active Withdrawal Check
		checks.push({ name: "Active Withdrawal" });
		const activeWithdrawals = await WithdrawalRequest.countDocuments({
			username: claim.username,
			status: {
				$in: ["pending", "processing", "validating", "transferring", "logging"],
			},
		});
		if (activeWithdrawals > 0) {
			result.canProceed = false;
			result.reason = "User has active withdrawal in progress";
			result.errors.push("Active withdrawal exists");
			result.riskLevel = "HIGH";
			return result;
		}
		result.checksPassedCount++;

		// 11. Claim Age Check
		checks.push({ name: "Claim Freshness" });
		const claimAgeHours =
			(Date.now() - new Date(claim.requestedAt || claim.createdAt).getTime()) /
			TIME_CONSTANTS.ONE_HOUR_MS;
		if (claimAgeHours > CLAIM_LIMITS.MAX_CLAIM_AGE_HOURS) {
			result.warnings.push(
				`Claim is ${Math.round(claimAgeHours)} hours old - stale data`,
			);
			if (claimAgeHours > 48) {
				result.riskLevel = "MEDIUM";
			}
		}
		result.checksPassedCount++;

		// 12. Claim History Check
		checks.push({ name: "Claim History" });
		const userClaimHistory = await ClaimRequest.find({
			username: claim.username,
			_id: { $ne: claim._id },
		})
			.sort({ createdAt: -1 })
			.limit(10)
			.lean();

		const rejectedClaims = userClaimHistory.filter(
			(c) => c.status === "rejected",
		).length;
		if (rejectedClaims > 2) {
			result.warnings.push(`User has ${rejectedClaims} rejected claims`);
			result.riskLevel = "HIGH";
		}
		result.checksPassedCount++;

		// 13. Claim Frequency Check
		checks.push({ name: "Claim Frequency" });
		const recentClaims = await ClaimRequest.countDocuments({
			username: claim.username,
			_id: { $ne: claim._id },
			createdAt: { $gte: new Date(Date.now() - TIME_CONSTANTS.ONE_DAY_MS) },
		});

		if (recentClaims > CLAIM_LIMITS.MAX_CLAIMS_PER_DAY) {
			result.warnings.push(
				`${recentClaims} claims submitted in last 24 hours - unusual activity`,
			);
			result.riskLevel = "HIGH";
		}
		result.checksPassedCount++;

		result.checksTotalCount = checks.length;

		logger.info(`[VERIFICATION SUCCESS] Pre-approval verification completed`, {
			claimId: claim._id,
			canProceed: result.canProceed,
			riskLevel: result.riskLevel,
			checksPassed: result.checksPassedCount,
			checksTotal: result.checksTotalCount,
			warningsCount: result.warnings.length,
			errorsCount: result.errors.length,
		});
	} catch (err) {
		logger.error(`[VERIFICATION ERROR] Pre-approval verification failed`, {
			claimId: claim._id,
			error: err.message,
			stack: err.stack,
		});
		logger.error("Pre-approval verification error:", {
			error: err.message,
			stack: err.stack,
			claimId: claim._id,
		});
		result.canProceed = false;
		result.reason = `Verification error: ${err.message}`;
		result.errors.push(err.message);
		result.riskLevel = "HIGH";
	}

	return result;
};

const processCleanupJob = async (job) => {
	const { data } = job;
	const { taskId, itemsToClean } = data;

	try {
		if (taskId) await taskService.markRunning(taskId);

		logger.info("Running data cleanup...", { itemsToClean });

		// Delete completed tasks older than 7 days
		const sevenDaysAgo = new Date(Date.now() - 7 * 24 * 60 * 60 * 1000);
		const deletedCount = await taskService.deleteOldCompleted(sevenDaysAgo);

		logger.info("Cleanup completed", { deletedCount });

		if (taskId) await taskService.markCompleted(taskId);
	} catch (err) {
		logger.error("Cleanup job failed", { error: err.message });
		if (taskId) await taskService.markFailed(taskId, err.message);
		throw err;
	}
};

// Scheduled job - runs every 5 minutes to check and queue cleanup
const scheduledDataCleanup = async () => {
	try {
		logger.debug("Checking for old completed tasks...");

		// Check for completed tasks older than 7 days
		const sevenDaysAgo = new Date(Date.now() - 7 * 24 * 60 * 60 * 1000);
		const oldTasksCount = await taskService.countOldCompleted(sevenDaysAgo);

		// Only queue cleanup job if there are old tasks to delete
		if (oldTasksCount > 0) {
			logger.info("Found old completed tasks, queuing cleanup job...", {
				oldTasksCount,
			});
			await taskService.create("data-cleanup", {
				timestamp: Date.now(),
				itemsToClean: oldTasksCount,
			});
		} else {
			logger.debug("No old tasks to cleanup");
		}
	} catch (err) {
		logger.error("Scheduled cleanup check failed", { error: err.message });
	}
};

const scheduledClaimApproval = async () => {
	logger.info(
		`[SCHEDULED CLAIM START] Starting scheduled claim approval check`,
	);

	try {
		// Skip if queue is paused - don't create new jobs
		if (queuePaused) {
			logger.debug(`CLAIM QUEUE | Queue is paused: ${queuePausedReason}`);
			return;
		}

		// Skip if already processing
		if (isProcessingClaim) {
			logger.debug(
				`CLAIM QUEUE | Already processing claim ${currentProcessingClaimId}`,
			);
			return;
		}

		const systemConfig = await AdminSystemSettings.findOne({
			type: "system_config",
		});
		const lastApprovalTime = systemConfig?.lastClaimApprovalTime
			? new Date(systemConfig.lastClaimApprovalTime).getTime()
			: 0;

		const now = Date.now();
		const timeSinceLastApproval = now - lastApprovalTime;
		if (
			timeSinceLastApproval < CLAIM_APPROVAL_INTERVAL_MS &&
			lastApprovalTime > 0
		) {
			const remainingSeconds = Math.ceil(
				(CLAIM_APPROVAL_INTERVAL_MS - timeSinceLastApproval) / 1000,
			);
			logger.debug(
				`CLAIM QUEUE | Cooldown active (30s between approvals), ${remainingSeconds}s remaining`,
			);
			return;
		}

		// Get next claim from queue (must have waited at least 1 minute)
		const oneMinuteAgo = new Date(now - CLAIM_MIN_WAIT_TIME_MS);
		const nextClaim = await ClaimRequest.findOne({
			status: "pending",
			queueStatus: { $in: ["waiting", "ready"] },
			processingLock: { $ne: true },
			requestedAt: { $lte: oneMinuteAgo },
		})
			.sort({ queuePosition: 1, requestedAt: 1 })
			.limit(1);

		if (!nextClaim) {
			logger.debug("CLAIM QUEUE | No claims ready for processing");
			return;
		}

		logger.info(`CLAIM QUEUE | Found claim ready for processing`, {
			claimId: nextClaim._id,
			username: nextClaim.username,
			amountNTE: nextClaim.amountNTE,
			queuePosition: nextClaim.queuePosition,
			queuedAt: nextClaim.queuedAt,
		});

		// Queue the approval job
		await taskService.create("claim-approval", {
			claimRequestId: nextClaim._id.toString(),
			username: nextClaim.username,
			amountNTE: nextClaim.amountNTE,
			timestamp: Date.now(),
		});

		logger.info(
			`[SCHEDULED CLAIM SUCCESS] Scheduled claim approval check completed`,
		);
	} catch (err) {
		logger.error(`[SCHEDULED CLAIM ERROR] Scheduled claim approval failed`, {
			error: err.message,
			stack: err.stack,
		});
	}
};

const updateQueuePositions = async () => {
	try {
		const updatedCount = await ClaimRequest.updateQueuePositions();
		if (updatedCount > 0) {
			logger.debug(`CLAIM QUEUE | Updated ${updatedCount} queue positions`);
		}
	} catch (err) {
		logger.error("CLAIM QUEUE | Position update error", { error: err.message });
	}
};

const cleanStaleLocks = async () => {
	logger.info(`[STALE LOCK CLEANUP START] Starting stale lock cleanup process`);

	try {
		const lockTimeout = LOCK_TIMEOUT_MS;
		const staleTime = new Date(Date.now() - lockTimeout);

		logger.info("STALE LOCK CLEANUP | Starting cleanup process...");

		const staleClaimLocks = await ClaimRequest.find({
			processingLock: true,
			processingLockedAt: { $lt: staleTime },
			status: "pending",
		}).lean();

		if (staleClaimLocks.length > 0) {
			logger.warn(
				`STALE LOCK CLEANUP | Found ${staleClaimLocks.length} stale claim locks`,
				{
					claims: staleClaimLocks.map((c) => ({
						id: c._id,
						lockedBy: c.processingLockedBy,
						lockedAt: c.processingLockedAt,
						ageMinutes: Math.floor(
							(Date.now() - new Date(c.processingLockedAt).getTime()) / 60000,
						),
						status: c.status,
					})),
				},
			);

			const claimCleanupResult = await ClaimRequest.updateMany(
				{
					processingLock: true,
					processingLockedAt: { $lt: staleTime },
					status: "pending",
					$or: [
						{ queueStatus: { $ne: "processing" } },
						{ queueStatus: { $exists: false } },
					],
				},
				{
					$set: {
						processingLock: false,
						processingLockedAt: null,
						processingLockedBy: null,
						queueStatus: "waiting",
					},
					$push: {
						autoApprovalErrorHistory: {
							attempt: 0,
							error: "Lock timed out and was released by stale lock cleanup",
							timestamp: new Date(),
						},
					},
				},
			);

			logger.info(
				`Released ${claimCleanupResult.modifiedCount} stale claim locks`,
			);

			for (const claim of staleClaimLocks) {
				await auditLogService.logSecurityViolation({
					type: "STALE_CLAIM_LOCK_CLEANUP",
					username: claim.username,
					description: `Stale processing lock cleaned for claim ${claim._id}. Was locked by ${claim.processingLockedBy} at ${claim.processingLockedAt}`,
					severity: "MEDIUM",
					clientIP: "system",
					userAgent: "stale-lock-cleanup",
					metadata: {
						claimId: claim._id,
						lockedBy: claim.processingLockedBy,
						lockedAt: claim.processingLockedAt,
						lockAgeMinutes: Math.floor(
							(Date.now() - new Date(claim.processingLockedAt).getTime()) /
								60000,
						),
					},
				});
			}
		}

		const distributedLockTimeout = DISTRIBUTED_LOCK_TIMEOUT_MS;
		const staleDistributedTime = new Date(
			Date.now() - distributedLockTimeout - 60000,
		);

		const staleDistributedLocks = await ClaimRequest.find({
			distributedLock: { $exists: true, $ne: null },
			distributedLockHeartbeat: { $lt: staleDistributedTime },
			status: "pending",
		}).lean();

		if (staleDistributedLocks.length > 0) {
			logger.warn(
				`STALE LOCK CLEANUP | Found ${staleDistributedLocks.length} stale distributed locks`,
			);

			const distributedCleanupResult = await ClaimRequest.updateMany(
				{
					distributedLock: { $exists: true, $ne: null },
					distributedLockHeartbeat: { $lt: staleDistributedTime },
					status: "pending",
				},
				{
					$unset: {
						distributedLock: "",
						distributedLockAcquiredAt: "",
						distributedLockExpiry: "",
						distributedLockWorker: "",
						distributedLockHeartbeat: "",
					},
				},
			);

			logger.info(
				`Released ${distributedCleanupResult.modifiedCount} stale distributed locks`,
			);
		}

		const staleBalanceLocks = await User.find({
			balanceLocked: true,
			balanceLockedAt: { $lt: staleTime },
		})
			.select(
				"_id username balanceLocked balanceLockedAt balanceLockedBy balanceLockedAmount",
			)
			.lean();

		if (staleBalanceLocks.length > 0) {
			logger.warn(
				`STALE LOCK CLEANUP | Found ${staleBalanceLocks.length} stale balance locks`,
				{
					users: staleBalanceLocks.map((u) => ({
						id: u._id,
						username: u.username,
						lockedBy: u.balanceLockedBy,
						lockedAt: u.balanceLockedAt,
						amount: u.balanceLockedAmount,
						ageMinutes: Math.floor(
							(Date.now() - new Date(u.balanceLockedAt).getTime()) / 60000,
						),
					})),
				},
			);

			const balanceCleanupResult = await User.updateMany(
				{
					balanceLocked: true,
					balanceLockedAt: { $lt: staleTime },
				},
				{
					$set: {
						balanceLocked: false,
						balanceLockedAt: null,
						balanceLockedBy: null,
						balanceLockedAmount: 0,
					},
				},
			);

			logger.info(
				`Released ${balanceCleanupResult.modifiedCount} stale balance locks`,
			);

			for (const user of staleBalanceLocks) {
				await auditLogService.logSecurityViolation({
					type: "STALE_BALANCE_LOCK_CLEANUP",
					username: user.username,
					description: `Stale balance lock cleaned for user ${user.username}. Was locked by ${user.balanceLockedBy} at ${user.balanceLockedAt} with amount ${user.balanceLockedAmount} NTE`,
					severity: "HIGH",
					clientIP: "system",
					userAgent: "stale-lock-cleanup",
					metadata: {
						userId: user._id,
						lockedBy: user.balanceLockedBy,
						lockedAt: user.balanceLockedAt,
						lockedAmount: user.balanceLockedAmount,
						lockAgeMinutes: Math.floor(
							(Date.now() - new Date(user.balanceLockedAt).getTime()) / 60000,
						),
					},
				});
			}
		}

		if (staleClaimLocks.length === 0 && staleBalanceLocks.length === 0) {
			logger.debug("No stale locks found");
		}

		logger.info(`[STALE LOCK CLEANUP SUCCESS] Stale lock cleanup completed`, {
			claimLocksReleased: staleClaimLocks.length || 0,
			balanceLocksReleased: staleBalanceLocks.length || 0,
		});
	} catch (err) {
		logger.error(`[STALE LOCK CLEANUP ERROR] Stale lock cleanup failed`, {
			error: err.message,
			stack: err.stack,
		});
	}
};

const pauseClaimQueue = (reason, pausedBy = "system") => {
	queuePaused = true;
	queuePausedReason = reason;
	queuePausedAt = new Date();
	logger.warn(`CLAIM QUEUE | Queue paused by ${pausedBy}: ${reason}`);
	return { paused: true, reason, pausedAt: queuePausedAt, pausedBy };
};

const resumeClaimQueue = async (resumedBy = "system") => {
	queuePaused = false;
	const previousReason = queuePausedReason;
	queuePausedReason = null;
	queuePausedAt = null;
	consecutiveFailures = 0;
	logger.info(
		`CLAIM QUEUE | Queue resumed by ${resumedBy}. Previous pause reason: ${previousReason}`,
	);

	// Reschedule all pending claims based on priority, then process one by one
	try {
		const pendingClaims = await ClaimRequest.find({
			status: "pending",
			queuedForApproval: true,
		})
			.sort({ priority: -1, queuePosition: 1, requestedAt: 1 }) // Sort by priority DESC, then queue position
			.lean();

		logger.info(
			`CLAIM QUEUE | Rescheduling ${pendingClaims.length} pending claims after resume`,
		);

		const now = Date.now();
		const CLAIM_MIN_WAIT_TIME_MS = 60 * 1000; // 1 minute for first claim
		const CLAIM_APPROVAL_INTERVAL_MS = 30 * 1000; // 30 seconds between claims

		// Update claims with new priority-based queue positions and timing
		for (let i = 0; i < pendingClaims.length; i++) {
			const claim = pendingClaims[i];
			const newQueuePosition = i + 1;

			// Calculate staggered timing - first claim in 1 min, then +30s for each subsequent
			const estimatedWaitTime =
				newQueuePosition === 1
					? CLAIM_MIN_WAIT_TIME_MS
					: CLAIM_MIN_WAIT_TIME_MS +
					  (newQueuePosition - 1) * CLAIM_APPROVAL_INTERVAL_MS;
			const estimatedApprovalTime = new Date(now + estimatedWaitTime);

			// Update claim with new queue position and timing
			await ClaimRequest.findByIdAndUpdate(claim._id, {
				$set: {
					queuePosition: newQueuePosition,
					queueStatus: "waiting", // Set to waiting, not ready
					scheduledProcessingTime: estimatedApprovalTime,
					estimatedApprovalTime,
				},
				$push: {
					queuePositionHistory: {
						position: newQueuePosition,
						estimatedTime: estimatedApprovalTime,
						updatedAt: new Date(),
					},
				},
			});
		}

		logger.info(
			`CLAIM QUEUE | Rescheduled ${pendingClaims.length} claims based on priority. Scheduler will process them one by one.`,
		);
	} catch (error) {
		logger.error(`CLAIM QUEUE | Error rescheduling claims on resume:`, error);
	}

	return { paused: false, resumedBy, previousReason };
};

const getQueueStatus = async () => {
	const systemConfig = await AdminSystemSettings.findOne({
		type: "system_config",
	});
	const lastApprovalTime = systemConfig?.lastClaimApprovalTime
		? new Date(systemConfig.lastClaimApprovalTime).getTime()
		: 0;

	return {
		paused: queuePaused,
		pausedReason: queuePausedReason,
		pausedAt: queuePausedAt,
		isProcessing: isProcessingClaim,
		currentProcessingClaimId,
		lastApprovalTime: lastApprovalTime > 0 ? new Date(lastApprovalTime) : null,
		lastApprovalBy: systemConfig?.lastClaimApprovalBy || null,
		consecutiveFailures,
		intervalMs: CLAIM_APPROVAL_INTERVAL_MS,
		nextAllowedProcessTime:
			lastApprovalTime > 0
				? new Date(lastApprovalTime + CLAIM_APPROVAL_INTERVAL_MS)
				: new Date(),
	};
};

const queueCleanupJob = async () => {
	return await taskService.create("data-cleanup", { timestamp: Date.now() });
};

// Process bulk email job
const processBulkEmailJob = async (job) => {
	const { data } = job;
	const {
		from,
		recipients,
		cc,
		bcc,
		subject,
		text,
		html,
		contentType,
		sentBy,
		batchSize,
		delayBetweenBatches,
		maxRetries,
		bulkEmailId,
	} = data;

	try {
		logger.info(`[BULK EMAIL JOB START] Processing bulk email`, {
			jobId: job.id,
			bulkEmailId,
			recipientCount: recipients?.length || 0,
			sentBy,
		});

		const result = await emailService.sendBulkEmail({
			from,
			recipients,
			cc,
			bcc,
			subject,
			text,
			html,
			contentType,
			sentBy,
			batchSize,
			delayBetweenBatches,
			maxRetries,
			bulkJobId: job.id,
			bulkEmailId,
		});

		logger.info(`[BULK EMAIL JOB SUCCESS] Bulk email completed`, {
			jobId: job.id,
			bulkEmailId,
			bulkId: result.bulkId,
			sentCount: result.sentCount,
			failedCount: result.failedCount,
			totalRecipients: result.totalRecipients,
			duration: result.duration,
			successRate:
				result.sentCount > 0
					? `${((result.sentCount / result.totalRecipients) * 100).toFixed(1)}%`
					: "0%",
		});

		return result;
	} catch (error) {
		logger.error(`[BULK EMAIL JOB ERROR] Failed to process bulk email`, {
			jobId: job.id,
			bulkEmailId,
			error: error.message,
			stack: error.stack,
		});
		throw error;
	}
};

// Process bulk push notification job
const processBulkPushNotificationJob = async (job) => {
	const { data } = job;
	const {
		payload,
		bulkNotificationId,
		batchSize,
		delayBetweenBatches,
		maxRetries,
		sentBy,
	} = data;

	try {
		logger.info(`[BULK PUSH JOB START] Processing bulk push notification`, {
			jobId: job.id,
			bulkNotificationId,
			payloadTitle: payload?.title,
			sentBy,
		});

		const result = await pushSubscriptionService.sendNotificationToAll(
			payload,
			{
				batchSize,
				delayBetweenBatches,
				maxRetries,
				bulkJobId: job.id,
				bulkNotificationId,
				sentBy,
			},
		);

		logger.info(`[BULK PUSH JOB SUCCESS] Bulk push notification completed`, {
			jobId: job.id,
			bulkNotificationId: result.bulkNotificationId,
			sentCount: result.sentCount,
			failedCount: result.failedCount,
			totalUsers: result.totalUsers,
			duration: result.duration,
			successRate:
				result.sentCount > 0
					? `${((result.sentCount / result.totalUsers) * 100).toFixed(1)}%`
					: "0%",
		});

		return result;
	} catch (error) {
		logger.error(
			`[BULK PUSH JOB ERROR] Failed to process bulk push notification`,
			{
				jobId: job.id,
				bulkNotificationId,
				error: error.message,
				stack: error.stack,
			},
		);
		throw error;
	}
};

// Start worker
const startWorker = async () => {
	try {
		// Check and log transaction support
		logger.info("Checking MongoDB transaction support...");
		const transactionsSupported = await checkTransactionSupport();

		// Register job processors
		registerProcessor("process-referral-bonuses", processReferralBonusesJob);
		registerProcessor("process-dynamic-pricing", processDynamicPricingJob);
		registerProcessor(
			"update-package-owners-cache",
			processPackageOwnersCacheJob,
		);
		registerProcessor("data-cleanup", processCleanupJob);
		registerProcessor("claim-approval", processClaimApprovalJob);
		registerProcessor("send-bulk-email", processBulkEmailJob);
		registerProcessor(
			"send-bulk-push-notification",
			processBulkPushNotificationJob,
		);

		// Register scheduled jobs
		schedulerService.register(
			"scheduled-cleanup",
			scheduledDataCleanup,
			SCHEDULER_INTERVAL_MS,
		);

		// Claim queue processor - checks every 30 seconds
		schedulerService.register(
			"scheduled-claim-approval",
			scheduledClaimApproval,
			QUEUE_CHECK_INTERVAL_MS,
		);

		// Queue position updater - runs every minute
		schedulerService.register(
			"queue-position-updater",
			updateQueuePositions,
			QUEUE_POSITION_UPDATE_INTERVAL_MS,
		);

		// Stale lock cleaner - runs every 5 minutes
		schedulerService.register(
			"stale-lock-cleaner",
			cleanStaleLocks,
			SCHEDULER_INTERVAL_MS,
		);

		logger.info("==================== WORKER STARTED ====================");
		logger.info(
			`Claim timing: 1 min wait from request, ${
				CLAIM_APPROVAL_INTERVAL_MS / 1000
			}s between approvals`,
		);
		logger.info(
			`Queue check frequency: every ${QUEUE_CHECK_INTERVAL_MS / 1000}s`,
		);
		logger.info(
			`Queue position update: every ${
				QUEUE_POSITION_UPDATE_INTERVAL_MS / 1000
			}s`,
		);
		logger.info(
			`Max consecutive failures before pause: ${MAX_CONSECUTIVE_FAILURES}`,
		);
		logger.info("=========================================================");
	} catch (err) {
		logger.error("Worker bootstrap failed", { error: err.message });
		throw err;
	}
};

// Stop worker
const stopWorker = async () => {
	logger.info("Stopping worker...");
	schedulerService.stopAll();
};

module.exports = {
	startWorker,
	stopWorker,
	queueCleanupJob,
	// Claim queue control exports for admin
	pauseClaimQueue,
	resumeClaimQueue,
	getQueueStatus,
};

// Standalone mode - run worker as separate process
if (require.main === module) {
	const shutdown = async (signal) => {
		logger.info(`${signal} received, worker shutting down...`);
		await stopWorker();

		try {
			await mongoose.connection.close();
			logger.info("MongoDB connection closed");
		} catch (err) {
			logger.error("Error closing MongoDB", { error: err.message });
		}

		process.exit(0);
	};

	process.on("SIGTERM", () => shutdown("SIGTERM"));
	process.on("SIGINT", () => shutdown("SIGINT"));
	process.on("uncaughtException", (err) => {
		logger.error("Worker uncaught exception", {
			error: err.message,
			stack: err.stack,
		});
		process.exit(1);
	});
	process.on("unhandledRejection", (reason) => {
		logger.error("Worker unhandled rejection", { reason: String(reason) });
		process.exit(1);
	});

	(async () => {
		try {
			await connectDb();
			await startWorker();
		} catch (err) {
			logger.error("Worker bootstrap failed", { error: err.message });
			process.exit(1);
		}
	})();
}
